/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#ifndef MACHINEIPPHEADERFILE
#define MACHINEIPPHEADERFILE

#include <QString>
#include <QTimer>
#include <vector>

#include "../../core/mat4.h"


#include "machine.h"

class QTimer;

class TMachineIpp : public TMachine
{
	Q_OBJECT
	
public:
	// CREATORS
	TMachineIpp(void);
	virtual ~TMachineIpp(void);
	
	// ACCESSORS
	virtual QString Machine_Description(void) const {return QStringLiteral("I++ Server");}
	
	// MANIPULATORS
	virtual void Process_Rx(const QByteArray &new_data);
	virtual QByteArray KeyPress_Done(void) const;
	virtual QByteArray KeyPress_DelPnt(void) const;

	virtual void Set_EStop(const bool state);
	virtual bool Move_To(const TVector3 &pos);
	virtual bool Manual_Touch(const TVector3 &pos,const TVector3 &vec);
	virtual bool Jog_Move(const TVector3 &direction);
	
private slots:
	void Position_Report_Time(void);
	void Position_Report_Distance(void);
	
private:
	
struct TToolData
{
	QString								name;
	QString								name_hint;
	TVector3							offset;
	TVector3							e_offset;
	TVector3							vector;
	double								head_a;
	double								head_b;
	double								tip_diameter;
};
	
struct TAlignmentData
{
	QString								name;
	bool								name_is_enum;
	TMat4								csy_mat;
	TVector3							csy_origin;
	double								csy_alpha;
	double								csy_beta;
	double								csy_gamma;
};
	
enum TDataProperty
{
	PROPERTY_X = 0,
	PROPERTY_Y,
	PROPERTY_Z,
	PROPERTY_TOOL_A,
	PROPERTY_TOOL_B,
	PROPERTY_IJK,
	PROPERTY_IJKACT,
	PROPERTY_ER,						// effective tip radius
	PROPERTY_Q,							// quality of measurement.  0->100, 0 = best
	PROPERTY_R							// rotary table rotation angle
};
	
	QTimer								*d_position_daemon;
	QString								d_position_daemon_command_tag;
	int									d_position_time;
	double								d_position_distance;
	TVector3							d_last_position;
	
	QByteArray							d_rx_data;
	std::vector<TToolData>				d_tools;
	TMachineIpp::TToolData				d_active_tool;
	TMachineIpp::TToolData				d_found_tool;
	bool								d_session_active;
	bool								d_error_active;
	bool								d_user_enabled;
	
	TAlignmentData						d_active_alignment;
	std::vector<TAlignmentData>			d_alignments;

	std::vector<TDataProperty>			d_position_report_properties;
	std::vector<TDataProperty>			d_pt_meas_report_properties;
	std::vector<TDataProperty>			d_scan_report_properties;
	
	bool Process_Line(const QString &line,QString * const tag,QString * const remainder);
	void Create_Tool_Data(void);
	void Start_Position_Report(const QString &command_tag,const QString &arguments);
	
	void Initialize_Alignment_Data(void);
	void Update_Alignment_Data(const TAlignmentData &alignment_data);				// adds if not found
	void Delete_Alignment_Data(const QString &name);
	TAlignmentData Find_Alignment(const QString &name,bool * const exists) const;	// name or enum_name.
	
	
	std::vector<TDataProperty> Get_Report_Properties(const QString &arguments) const;
	std::vector<double> Get_Argument_Values(const QString &arguments) const;

	QString Format_Property(const std::vector<TDataProperty> &properties,
						const TVector3 &pos,
						const TVector3 &vec = TVector3(0,0,1)) const;	// information such as tool angles, tip radius, ... read directly from base class
	
	bool Get_Move_Data(const QString &arguments,TVector3 * const pos) const;
	bool Get_Touch_Data(const QString &arguments,TVector3 * const pos,TVector3 * const ijk) const;
	TToolData Find_Tool(const QString &name) const;
	TToolData Interpolate_Tool(const double &head_a,const double &head_b) const;
	TVector3 Convert_Pos_To_Part_Csy(const TVector3 &pos) const;
	TVector3 Convert_Vector_To_Part_Csy(const TVector3 &vec) const;
	TVector3 Convert_Pos_From_Part_Csy(const TVector3 &pos) const;
	TVector3 Convert_Vector_From_Part_Csy(const TVector3 &vec) const;
	
	void Set_Active_Tool_EOffset(const TVector3 &e_offset);

	// NOT IMPLEMENTED
	TMachineIpp(const TMachineIpp&);
	TMachineIpp& operator=(const TMachineIpp&);
};

#endif
