/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QTimer>

#include "machine_leitz.h"

static const int						TIMER_INTERVAL(40);

TMachineLeitz::TMachineLeitz(void)
{
	d_geo_enabled = true;
	d_shift_out_enabled = false;
}

TMachineLeitz::~TMachineLeitz(void)
{
}

void TMachineLeitz::Process_Rx(
	const QByteArray					&new_data)
{
	QByteArray							byte_sequence;
	int									rx_line_index(-1);
	int									rx_byte_index(-1);
	int									index;
	QStringList							list;
	QString								text;
	QString								write_text;
	QString								line;
	QString								keyword;
	QString								arguments;
	TVector3							pos;
	TVector3							vec;
	TVector3							p1,p2;
	double								a,b;
	double								dval;
	bool								x_error,y_error,z_error;
	int									arg1,arg2,arg3,arg4,arg5,arg6;
	TCommandExecutor::TCommand			command;
	
	d_rx_data += new_data;
	
	if(d_shift_out_enabled)
	{
		this->Send_Clean(new_data);
	}

	do
	{
		if(d_rx_data.size())
		{
			
#if DEBUG_MODE_LEVEL_0
			QByteArray 	control_bytes;
			int			control_byte_cntr;
			char		byte;
			
			for(control_byte_cntr = 0;control_byte_cntr < d_rx_data.size();++control_byte_cntr)
			{
				byte = d_rx_data[control_byte_cntr];
				
				if(byte < 0x20 &&
				   byte != '\r' &&
				   byte != '\n')
				{
					control_bytes.push_back(byte);
				}
			}
			
			if(control_bytes.size())
			{
				qWarning(QString("RECV: %1").arg(QString(control_bytes.toHex())).toLatin1());
			}
#endif
			
			rx_byte_index = d_rx_data.indexOf(char(0x02));
			
			if(!(rx_byte_index < 0))
			{
				d_rx_data.remove(rx_byte_index,1);
				
#if DEBUG_MODE_LEVEL_1
				qWarning(QString("RECV: CTRL_B (0x02)").toLatin1());
#endif
				d_command_executor.Abort();
				
				d_jog_timer->stop();
				d_control_timer->stop();
				
				this->Send_Delayed("READY\r\n",50);
			}
			
			rx_byte_index = d_rx_data.indexOf(char(0x03));
			
			if(!(rx_byte_index < 0))
			{
				d_rx_data.remove(rx_byte_index,1);
				
#if DEBUG_MODE_LEVEL_1
				qWarning(QString("RECV: CTRL_C (0x03)").toLatin1());
#endif
				d_command_executor.Abort();
				
				d_jog_timer->stop();
				d_control_timer->stop();
			}
			
			rx_byte_index = d_rx_data.indexOf(char(0x05));
			
			if(!(rx_byte_index < 0))
			{
				d_rx_data.remove(rx_byte_index,1);
				d_shift_out_enabled = true;
				
#if DEBUG_MODE_LEVEL_1
				qWarning(QString("RECV: CTRL_E (0x05)").toLatin1());
#endif
				d_command_executor.Abort();
				
				d_jog_timer->stop();
				d_control_timer->stop();
			}
			
			rx_byte_index = d_rx_data.indexOf(char(0x08));
			
			if(!(rx_byte_index < 0))
			{
				byte_sequence.resize(3);
				//				byte_sequence[0] = char(0x1B);
				//				byte_sequence[1] = char(0x7f);
				byte_sequence[0] = char(0x08);
				byte_sequence[1] = char(0x20);
				byte_sequence[2] = char(0x08);
				
				this->Send(byte_sequence);
				
				d_rx_data.remove(rx_byte_index,1);
				d_rx_data.chop(1);		// remove last byte
				
#if DEBUG_MODE_LEVEL_1
				qWarning(QString("RECV: CTRL_H (0x08) Backspace").toLatin1());
#endif
			}
			
			rx_byte_index = d_rx_data.indexOf(char(0x0e));
			
			if(!(rx_byte_index < 0))
			{
				d_rx_data.remove(rx_byte_index,1);
				d_shift_out_enabled = false;
				
#if DEBUG_MODE_LEVEL_1
				qWarning(QString("RECV: CTRL_N (0x0e)").toLatin1());
#endif
			}
			
			rx_byte_index = d_rx_data.indexOf(char(0x1a));
			
			if(!(rx_byte_index < 0))
			{
				d_rx_data.remove(rx_byte_index,1);
				d_shift_out_enabled = false;
				
#if DEBUG_MODE_LEVEL_1
				qWarning(QString("RECV: CTRL_Z (0x1a)").toLatin1());
#endif
				d_command_executor.Abort();
				
				d_jog_timer->stop();
				d_control_timer->stop();
				
				this->Send("READY\r\n");
			}
		}
	}while(!(rx_byte_index < 0) && d_rx_data.size());
	
	do
	{
		rx_line_index = d_rx_data.indexOf('\r');
		
		if(!(rx_line_index < 0))
		{
			line = QString::fromLatin1(d_rx_data.mid(0,rx_line_index)).simplified();
			d_rx_data.remove(0,rx_line_index + 1);
			
			// clean text
			while(line.length() && line[0].toLatin1() < 0x20)
			{
				line = line.mid(1);
			}
			
			emit Log_Rx(line);
			
			if(d_shift_out_enabled)
			{
#if DEBUG_MODE_LEVEL_1
				qWarning(QString("INFO:  Shift Out Enabled").toLatin1());
#endif
				this->Send("\r\n");
			}
			
			index = line.indexOf(' ');
			
			if(!(index < 0))
			{
				keyword = line.mid(0,index).toUpper();
				arguments = line.mid(index + 1);
			}
			else
			{
				keyword = line.toUpper();
				arguments = QString();
			}
			
#if DEBUG_MODE_LEVEL_1
			qWarning(QString("RECV: %1").arg(line).toLatin1());
#endif
			
			if(keyword.compare(QStringLiteral("ACCEL")) == 0)
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("AUTZER")) == 0)
			{
				if(d_estop_status)
				{
					this->Send("ERROR MOTOR_RE,S%1,TR_FG013,E-Stop Push-Button Pressed\r\n");
				}
				else
				{
					pos = d_command_executor.Current_Position();
					p1 = d_min_machine;
					p2 = d_max_machine;
					
					p1.z = p2.z;
					
					command.command_type = TCommandExecutor::COMMAND_MOVE;
					command.target = TVector3(pos.x,pos.y,p1.z + d_tool_offset.z);
					command.speed = d_move_speed;
					command.completion_text = QString();
					
					d_command_executor.Add_Command(command);
					
					command.command_type = TCommandExecutor::COMMAND_MOVE;
					command.target = p1 + d_tool_offset;
					command.speed = d_move_speed;
					command.completion_text = QStringLiteral("%\r\n");
					
					d_command_executor.Add_Command(command);
					
					if(!d_control_timer->isActive())
					{
						d_control_timer->start(TIMER_INTERVAL);
					}
					
					d_homed_status = true;
				}
			}
			else if(keyword.compare(QStringLiteral("CMHWPF")) == 0)
			{
				this->Send("CMHWPF            4,        99,         4\r\n");
			}
			else if(keyword.compare(QStringLiteral("CMSWPF")) == 0)
			{
				/*
				 Argument 1               Argument 2              Argument 3              Argument 4              Argument 5              Argument 6
				 0 CMM_KNOWS_CMMTYP     	 0 CMM_KNOWS_ACCEL   	 0 CMM_KNOWS_SCADAT  	 0 CMM_KNOWS_CMMACSY  	 0 CMM_KNOWS_SCLTMP  	 0 CMM_KNOWS_VMABRT
				 1 CMM_KNOWS_AUTZER     	 1 CMM_KNOWS_MOVABS  	 1 CMM_KNOWS_SCADIR  	 1 CMM_KNOWS_OFSTCSY  	 1 CMM_KNOWS_WKPTCSY 	 1 CMM_KNOWS_STATUS
				 2 CMM_KNOWS_AUTORP     	 2 CMM_KNOWS_MOVDLT  	 2 CMM_KNOWS_SCAINT  	 2 CMM_KNOWS_REFRCSY  	 2 CMM_KNOWS_WKPPAR  	 2 CMM_KNOWS_CANCEL
				 3 CMM_KNOWS_INITCM     	 3 CMM_KNOWS_MOVPAR  	 3 CMM_KNOWS_SCAN    	 3 CMM_KNOWS_RFIXCSY  	 3 CMM_KNOWS_PRBTPA  	 3 CMM_KNOWS_DUMP
				 4 CMM_KNOWS_RESET      	 4 CMM_KNOWS_GETPOS  	 4 CMM_KNOWS_SCAPNT  	 4 CMM_KNOWS_ROFSCSY  	 4 CMM_KNOWS_READTP  	 4 CMM_KNOWS_DEVFUN
				 5 CMM_KNOWS_DISABLE    	 5 CMM_KNOWS_GETDVM  	 5 CMM_KNOWS_SCASTA  	 5 CMM_KNOWS_RPHICSY  	 5 CMM_KNOWS_TWARNING	 5 CMM_KNOWS_CADATA
				 6 CMM_KNOWS_ENABLE     	 6 CMM_KNOWS_RPTPOS  	 6 CMM_KNOWS_SCASTO  	 8 CMM_KNOWS_JOYSTCSY 	16 CMM_KNOWS_INIMAC  	 6 CMM_KNOWS_AUTCAL
				 7 CMM_KNOWS_SHOW       	 7 CMM_KNOWS_GETPOSM 	 7 CMM_KNOWS_HSSDAT  	 9 CMM_KNOWS_DISPCSY  	17 CMM_KNOWS_DFNMAC  	 7 CMM_KNOWS_PSTOVM
				 8 CMM_KNOWS_VERSION    	16 CMM_KNOWS_PRBRPT  	 8 CMM_KNOWS_HSSAXI  	10 CMM_KNOWS_DSPINF   	18 CMM_KNOWS_EDTMAC  	 8 CMM_KNOWS_DEFINE
				 9 CMM_KNOWS_CMHWPF     	17 CMM_KNOWS_PRBATT  	 9 CMM_KNOWS_HSSCIR  	11 CMM_KNOWS_DIMENS   	19 CMM_KNOWS_USEMAC  	 9 CMM_KNOWS_XEFILE
				 10 CMM_KNOWS_CMHWST     18 CMM_KNOWS_PRBLPA  	10 CMM_KNOWS_HSSCYL  	12 CMM_KNOWS_SETMSK   	20 CMM_KNOWS_MPRBFS  	10 CMM_KNOWS_SET
				 11 CMM_KNOWS_CMSWPF     19 CMM_KNOWS_PRBGPA  	16 CMM_KNOWS_CENDAT  	16 CMM_KNOWS_MAGACSY  	21 CMM_KNOWS_SETOFS  	11 CMM_KNOWS_HELP
				 16 CMM_KNOWS_BCKTRK     20 CMM_KNOWS_PRBHTY  	17 CMM_KNOWS_CENSTA  	17 CMM_KNOWS_NEXTPRB                            12 CMM_KNOWS_VMTEST
				 17 CMM_KNOWS_IBOPEN     21 CMM_KNOWS_PRBTYP  	18 CMM_KNOWS_CENTER  	18 CMM_KNOWS_SETMAGA                            13 CMM_KNOWS_ASCEFC
				 18 CMM_KNOWS_MOVTST     22 CMM_KNOWS_PRBPIN  	21 CMM_KNOWS_DPS_DAT 	24 CMM_KNOWS_PARK                               14 CMM_KNOWS_VMFRMT
				 19 CMM_KNOWS_QSDEV      23 CMM_KNOWS_PROBE   	22 CMM_KNOWS_DPS_AXI 	25 CMM_KNOWS_ENDPARK                            15 CMM_KNOWS_VMEXIT
				 20 CMM_KNOWS_PRBDATA    24 CMM_KNOWS_PRBRDV  	23 CMM_KNOWS_DPS_CIR                                                    16 CMM_KNOWS_SETSNSR
				 21 CMM_KNOWS_PRESSURE                           24 CMM_KNOWS_DPS_CYL                                                    17 CMM_KNOWS_PRBXPT
				 22 CMM_KNOWS_PRBTST                                                                                                     19 CMM_KNOWS_FLY
				 23 CMM_KNOWS_READVU                                                                                                     20 CMM_KNOWS_HSS-TAG
				 24 CMM_KNOWS_TESTCORR                                                                                                   21 CMM_KNOWS_VHSS
				 25 CMM_KNOWS_NEWDOWN
				 */
				
				arg1 = 0b00000000000000000000111011111011;
				arg2 = 0b00000000111111110000000000011011;
				arg3 = 0b00000000000000000000000000000000;
				arg4 = 0b00000000000000000000001001000000;
				arg5 = 0b00000000000000000000000000010100;
				arg6 = 0b00000000000010000000010000000010;
				
				write_text = QString("CMSWPF %1, %2, %3, %4, %5, %6\r\n").arg(arg1,9).arg(arg2,9).arg(arg3,9).arg(arg4,9).arg(arg5,9).arg(arg6,9);
				
				this->Send(write_text.toLatin1());
			}
			else if(keyword.compare(QStringLiteral("CMHWST")) == 0)
			{
				/*
				 Argument 1                       Argument 2                          Argument 3
				 0 IS_ZEROED                   	 0 CMM_PART_TEMPCOMP_ENABLED      	 0 CMM_RFIXCSY_ENABLED
				 1 VOLCOMP_ENABLED             	 1 CMM_PROBE_TEMPCOMP_ENABLED     	 1 CMM_ROFSCSY_ENABLED
				 2 PROBE_REF_POINT_SET         	 2 CMM_TEMPCOMP_ENABLED           	 2 CMM_RPHICSY_ENABLED
				 3 PROBE_CHANGER_CSY_LOADED    	 3 CMM_ROTARY_TABLE_MOVE_ENABLED  	 3 CMM_OFSTCSY_ENABLED
				 4 ROTARY_TABLE_CALIBRATED     	 4 CMM_RTT_COMP_ENABLED           	 4 CMM_REFRCSY_ENABLED
				 5 DEFAULT_PARAMS_MODIFIED     	 5 CMM_GEO_COMP_ENABLED           	 5 CMM_CMMACSY_ENABLED
				 6 READY_TO_MOVE                                                      6 CMM_JOYSTCSY_ENABLED
				 7 QDS_OPEN                                                           7 CMM_DISPCSY_ENABLED
				 8 QDS_READY                                                          8 CMM_SENDS_PROBING_DIRECTION
				 9 QDS_PROBE_ACTIVE                                                   9 CMM_MOVE_VOLCOMPED
				 10 QDS_RTCSY_FOUND                                                   10 CMM_JOGBOX_ENABLED
				 11 QDS_PCCSY_FOUND                                                   11 CMM_USES_LARGE_BUFFER
				 12 PRB_CHANGER_PORTS_VALID                                           12 CMM_PROBE_IS_IN_HEAD
				 13 MOTORS_POWERED_UP
				 */
				
				if(d_homed_status)
				{
					arg1 = 0b0010000001000111;
				}
				else
				{
					arg1 = 0b0010000001000110;
				}

				arg2 = 0b0000000000100000;
				arg3 = 0b0000011000000000;
				
				write_text = QString("CMHWST %1, %2, %3\r\n").arg(arg1,9).arg(arg2,9).arg(arg3,9);
				
				this->Send(write_text.toLatin1());
			}
			else if(keyword.compare(QStringLiteral("CMMTYP")) == 0)
			{
				this->Send("CMMTYP   MA  1234, COM-CTL   V01.00, DMP 12 22 10    , 0\r\n");
			}
			else if(keyword.compare(QStringLiteral("DIMENS")) == 0)		// DIMENS IN or DIMENS MM
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("DISABLE")) == 0)
			{
				if(arguments.startsWith(QStringLiteral("GEO"),Qt::CaseInsensitive))
				{
					d_geo_enabled = false;
				}
				
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("ENABLE")) == 0)
			{
				if(arguments.startsWith(QStringLiteral("GEO"),Qt::CaseInsensitive))
				{
					d_geo_enabled = true;
				}
				
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("ESTATUS")) == 0)
			{
				if(d_estop_status)
				{
					this->Send("ESTOP\r\n%\r\n");
				}
				else
				{
					this->Send("ONLIN\r\n%\r\n");
				}
			}
			else if(keyword.compare(QStringLiteral("GETPOS")) == 0)
			{
				command.command_type = TCommandExecutor::COMMAND_GETPOS;
				command.completion_text = QStringLiteral("POS %1, %2, %3, 0.00000, 0.00000, 0.00000, 0.00000\r\n");
				
				d_command_executor.Add_Command(command);
				
				if(!d_control_timer->isActive())
				{
					d_control_timer->start(TIMER_INTERVAL);
				}

//				pos = d_command_executor.Current_Position();
//				write_text = QString("POS %1, %2, %3, 0.00000, 0.00000, 0.00000, 0.00000\r\n").arg(pos.x,0,'f',5).arg(pos.y,0,'f',5).arg(pos.z,0,'f',5);
//				this->Send(write_text.toLatin1());
			}
			else if(keyword.compare(QStringLiteral("INITCM")) == 0)
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("JOGPBCSY")) == 0)
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("JOYSTCSY")) == 0)
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("MOVPAR")) == 0)		// MOVPAR 500.000000,500.000000,500.000000
			{
				d_move_speed = this->Get_Move_Speed(arguments);
				
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("MOVABS")) == 0)		// MOVABS 500.000000,1000.000000,-600.000000
			{
				if(d_estop_status)
				{
					this->Send("ERROR MOTOR_RE,S%1,TR_FG013,E-Stop Push-Button Pressed\r\n");
				}
				else
				{
					pos = this->Get_Move_Data(arguments);
					
					if(this->Is_Position_Inside_Volume(pos,&x_error,&y_error,&z_error))
					{
						command.command_type = TCommandExecutor::COMMAND_MOVE;
						command.target = pos;
						command.speed = d_move_speed;
						command.completion_text = QStringLiteral("%\r\n");
						
						d_command_executor.Add_Command(command);
						
						if(!d_control_timer->isActive())
						{
							d_control_timer->start(TIMER_INTERVAL);
						}
					}
					else
					{
						if(x_error)
						{
							emit Log_Text(QStringLiteral("ERROR <X_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!"));
							this->Send("ERROR <X_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!\r\n");
							d_shift_out_enabled = false;
						}
						
						if(y_error)
						{
							emit Log_Text(QStringLiteral("ERROR <Y_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!"));
							this->Send("ERROR <Y_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!\r\n");
							d_shift_out_enabled = false;
						}
						
						if(z_error)
						{
							emit Log_Text(QStringLiteral("ERROR <Z_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!"));
							this->Send("ERROR <Z_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!\r\n");
							d_shift_out_enabled = false;
						}
						
						d_command_executor.Abort();
						
						d_jog_timer->stop();
						d_control_timer->stop();
					}
				}
			}
			else if(keyword.compare(QStringLiteral("MOVCIR")) == 0)		// MOVCIR 500.000000,1000.000000,-600.000000
			{
				if(d_estop_status)
				{
					this->Send("ERROR MOTOR_RE,S%1,TR_FG013,E-Stop Push-Button Pressed\r\n");
				}
				else
				{
					pos = this->Get_Move_Data(arguments);
					
					if(this->Is_Position_Inside_Volume(pos,&x_error,&y_error,&z_error))
					{
						command.command_type = TCommandExecutor::COMMAND_MOVECIR;
						command.target = pos;
						command.speed = d_move_speed;
						command.completion_text = QStringLiteral("%\r\n");
						
						d_command_executor.Add_Command(command);
						
						if(!d_control_timer->isActive())
						{
							d_control_timer->start(TIMER_INTERVAL);
						}
					}
					else
					{
						if(x_error)
						{
							emit Log_Text(QStringLiteral("ERROR <X_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!"));
							this->Send("ERROR <X_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!\r\n");
							d_shift_out_enabled = false;
						}
						
						if(y_error)
						{
							emit Log_Text(QStringLiteral("ERROR <Y_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!"));
							this->Send("ERROR <Y_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!\r\n");
							d_shift_out_enabled = false;
						}
						
						if(z_error)
						{
							emit Log_Text(QStringLiteral("ERROR <Z_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!"));
							this->Send("ERROR <Z_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!\r\n");
							d_shift_out_enabled = false;
						}
						
						d_command_executor.Abort();
						
						d_jog_timer->stop();
						d_control_timer->stop();
					}
				}
			}
			else if(keyword.compare(QStringLiteral("MOVCTR")) == 0)		// center position of MOVCIR.  MOVCTR 500.000000,1000.000000,-600.000000
			{
				if(d_estop_status)
				{
					this->Send("ERROR MOTOR_RE,S%1,TR_FG013,E-Stop Push-Button Pressed\r\n");
				}
				else
				{
					pos = this->Get_Move_Data(arguments);
					
					if(this->Is_Position_Inside_Volume(pos,&x_error,&y_error,&z_error))
					{
						command.command_type = TCommandExecutor::COMMAND_DEFINE_MOVECIR_CENTER;
						command.target = pos;
						command.completion_text = QStringLiteral("%\r\n");
						
						d_command_executor.Add_Command(command);
						
						if(!d_control_timer->isActive())
						{
							d_control_timer->start(TIMER_INTERVAL);
						}
					}
					else
					{
						if(x_error)
						{
							emit Log_Text(QStringLiteral("ERROR <X_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!"));
							this->Send("ERROR <X_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!\r\n");
							d_shift_out_enabled = false;
						}
						
						if(y_error)
						{
							emit Log_Text(QStringLiteral("ERROR <Y_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!"));
							this->Send("ERROR <Y_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!\r\n");
							d_shift_out_enabled = false;
						}
						
						if(z_error)
						{
							emit Log_Text(QStringLiteral("ERROR <Z_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!"));
							this->Send("ERROR <Z_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!\r\n");
							d_shift_out_enabled = false;
						}
						
						d_command_executor.Abort();
						
						d_jog_timer->stop();
						d_control_timer->stop();
					}
				}
			}
			else if(keyword.compare(QStringLiteral("PRBRPT")) == 0)
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("PRBGPA")) == 0)		// search distance, probe mode, max force
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("PRBLPA")) == 0)		// touch speed, probing accuracy, prehit/retract.
			{
				this->Update_Touch_Parameters(arguments);
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("PRBPIN")) == 0)		// PRBPIN X,Y,Z or PRBPIN X,Y,Z,M1,M2,M3,M4,M5,M6,M7,M8,M9,TIP_DIAM,LEN
			{
				p1 = d_tool_offset;
				this->Update_Tool_Data(arguments);
				
				// takes into account the shift in position from a change in the tool offset
				pos = d_command_executor.Current_Position();
				pos += (d_tool_offset - p1);
				
				d_command_executor.Set_Current_Position(pos);
				
				emit Tool_Offset_Changed(d_tool_offset.x,d_tool_offset.y,d_tool_offset.z,d_active_tip_radius * 2.0);
				emit Position_Changed(pos.x,pos.y,pos.z);
				
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("PRBHTY")) == 0)		// PRBHTY PH9,0.0,0.0
			{
				if(arguments.startsWith(QStringLiteral("PH9"),Qt::CaseInsensitive))
				{
					if(Get_Head_Angles(arguments,&a,&b))
					{
						d_head_angle_a = a;
						d_head_angle_b = b;
						
						emit Head_AB_Changed(a,b);
						
					}
				}
				
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("PRBTYP")) == 0)		// PRBTYP TP6
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("PROBE")) == 0)		// PROBE X,Y,Z,W,I,J,K
			{
				if(d_estop_status)
				{
					this->Send("ERROR MOTOR_RE,S%1,TR_FG013,E-Stop Push-Button Pressed\r\n");
				}
				else
				{
					if(this->Get_Touch_Data(arguments,&pos,&vec))
					{
						if(this->Is_Position_Inside_Volume(pos,&x_error,&y_error,&z_error))
						{
							p1 = pos + d_prehit_distance * vec + d_active_tip_radius * vec;
							
							command.command_type = TCommandExecutor::COMMAND_MOVE;
							command.target = p1;
							command.speed = d_move_speed;
							command.completion_text = QString();
							
							d_command_executor.Add_Command(command);
							
							p2 = pos + d_active_tip_radius * vec + this->Get_Noise_Offset(vec);
							
							command.command_type = TCommandExecutor::COMMAND_TOUCH;
							command.target = pos;
							command.speed = d_touch_speed;
							command.completion_text = QString("POINT %1, %2, %3, 0.00000, %4, %5, %6\r\n")
										.arg(p2.x,0,'f',5)
										.arg(p2.y,0,'f',5)
										.arg(p2.z,0,'f',5)
										.arg(vec.i,0,'f',5)
										.arg(vec.j,0,'f',5)
										.arg(vec.k,0,'f',5);
										
							d_command_executor.Add_Command(command);
							
							command.command_type = TCommandExecutor::COMMAND_MOVE;
							command.target = p1;
							command.speed = d_move_speed;
							command.completion_text = QString();
							
							d_command_executor.Add_Command(command);
							
							if(!d_control_timer->isActive())
							{
								d_control_timer->start(TIMER_INTERVAL);
							}
						}
						else
						{
							if(x_error)
							{
								emit Log_Text(QStringLiteral("ERROR <X_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!"));
								this->Send("ERROR <X_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!\r\n");
								d_shift_out_enabled = false;
							}
							
							if(y_error)
							{
								emit Log_Text(QStringLiteral("ERROR <Y_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!"));
								this->Send("ERROR <Y_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!\r\n");
								d_shift_out_enabled = false;
							}
							
							if(z_error)
							{
								emit Log_Text(QStringLiteral("ERROR <Z_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!"));
								this->Send("ERROR <Z_LIMIT>,S%1,TR_OUTOL,Machine Parameter out of tolerance!\r\n");
								d_shift_out_enabled = false;
							}
							
							d_command_executor.Abort();
							
							d_jog_timer->stop();
							d_control_timer->stop();
						}
					}
				}
			}
			else if(keyword.compare(QStringLiteral("READTP")) == 0)
			{
				index = arguments.toInt();
				
				switch(index)
				{
					case 1:
					case 2:
						dval = d_temperature_x;
						break;
						
					case 3:
					case 4:
						dval = d_temperature_y;
						break;
						
					case 5:
					case 6:
						dval = d_temperature_z;
						break;
						
					case 7:
						dval = d_temperature_part;
						break;
						
					default:
						dval = 20.0;
						break;
				}
				
				this->Send(QString("READTP      %1\r\n").arg(dval,0,'f',5).toLatin1());
			}
			else if(keyword.compare(QStringLiteral("REFRCSY")) == 0)
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("RESET")) == 0)
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("SCLTMP")) == 0)	// command may not have trailing parameters
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("SET_PAR")) == 0)
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("SHOW")) == 0)
			{
				if(arguments.compare(QStringLiteral("ACCEL"),Qt::CaseInsensitive) == 0)
				{
					this->Send("ACCEL   1000.00000,  1000.00000,  1000.00000,  1000.00000,  1000.00000\r\n");
				}
				else if(arguments.compare(QStringLiteral("GEO"),Qt::CaseInsensitive) == 0)	// SHOW GEO may not be a leitz command
				{
					if(d_geo_enabled)
					{
						this->Send("ENABLED\r\n");
					}
					else
					{
						this->Send("DISABLED\r\n");
					}
				}
				else if(arguments.compare(QStringLiteral("FLYPARSW"),Qt::CaseInsensitive) == 0)
				{
					this->Send("FLYPARSW     97\r\n");
				}
				else if(arguments.compare(QStringLiteral("HAVE_PH9"),Qt::CaseInsensitive) == 0)
				{
					this->Send("HAVE_PH9 1\r\n");
				}
				else if(arguments.compare(QStringLiteral("MAX_ACCL"),Qt::CaseInsensitive) == 0)
				{
					this->Send("MAX_ACCL   1000.00000,  1000.00000,  1000.00000,  1000.00000,  1000.00000\r\n");
				}
				else if(arguments.compare(QStringLiteral("MOVPAR"),Qt::CaseInsensitive) == 0)
				{
					write_text = QString("MOVPAR %1, %2, %3, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000\r\n")
					.arg(d_move_speed,0,'f',5)
					.arg(d_move_speed,0,'f',5)
					.arg(d_move_speed,0,'f',5);
					
					this->Send(write_text.toLatin1());
				}
				else if(arguments.compare(QStringLiteral("P_SWLIM"),Qt::CaseInsensitive) == 0)
				{
					pos = d_max_machine;
					
					write_text = QString("P_SWLIM  %1, %2, %3, 1000.00000, 1000.00000, 1000.00000\r\n").arg(pos.x,11,'f',5).arg(pos.y,11,'f',5).arg(pos.z,11,'f',5);
					
					this->Send(write_text.toLatin1());
				}
				else if(arguments.compare(QStringLiteral("PRBCONFG"),Qt::CaseInsensitive) == 0)
				{
					this->Send("PRBCONFG    0\r\n");
				}
				else if(arguments.compare(QStringLiteral("PRBATT"),Qt::CaseInsensitive) == 0)
				{
					this->Send("PRBATT  0.05000, 0.02500, 0.07500,      10, 1.00000\r\n");
				}
				else if(arguments.compare(QStringLiteral("PRBGPA"),Qt::CaseInsensitive) == 0)
				{
					this->Send("PRBGPA  2.50000,      0, 2.50000\r\n");
				}
				else if(arguments.compare(QStringLiteral("PRBHTY"),Qt::CaseInsensitive) == 0)
				{
					write_text = QString("PRBHTY  PH9     ,     %1,     %2\r\n").arg(d_head_angle_a,0,'f',5).arg(d_head_angle_b,0,'f',5);
					this->Send(write_text.toLatin1());
				}
				else if(arguments.compare(QStringLiteral("PRBLPA"),Qt::CaseInsensitive) == 0)
				{
					write_text = QString("PRBLPA %1,     0, %2\r\n").arg(d_touch_speed,11,'f',5).arg(d_prehit_distance,11,'f',5);
					this->Send(write_text.toLatin1());
				}
				else if(arguments.compare(QStringLiteral("PRBPIN"),Qt::CaseInsensitive) == 0)
				{
					write_text = QString("PRBPIN %1, %2, %3,\r\n       0.000000, 0.000000, 0.000000,\r\n       0.000000, 0.000000, 0.000000,\r\n       0.000000, 0.000000, 0.000000,\r\n       %4, -1\r\n%\r\n")
					.arg(d_tool_offset.x,11,'f',5)
					.arg(d_tool_offset.y,11,'f',5)
					.arg(d_tool_offset.z,11,'f',5)
					.arg(d_active_tip_radius * 2.0,0,'f',5);
					
					this->Send(write_text.toLatin1());
				}
				else if(arguments.compare(QStringLiteral("PRT_SENS"),Qt::CaseInsensitive) == 0)
				{
					this->Send("PRT_SENS      7\r\n");
				}
				else if(arguments.compare(QStringLiteral("M_SWLIM"),Qt::CaseInsensitive) == 0)
				{
					pos = d_min_machine;
					
					write_text = QString("M_SWLIM  %1, %2, %3, -1000.00000, -1000.00000, -1000.00000\r\n").arg(pos.x,11,'f',5).arg(pos.y,11,'f',5).arg(pos.z,11,'f',5);
					
					this->Send(write_text.toLatin1());
				}
				else if(arguments.compare(QStringLiteral("NCE_XSCL"),Qt::CaseInsensitive) == 0)
				{
					this->Send("NCE_XSCL      0.000010\r\n");
				}
				else if(arguments.compare(QStringLiteral("NCE_YSCL"),Qt::CaseInsensitive) == 0)
				{
					this->Send("NCE_YSCL      0.000010\r\n");
				}
				else if(arguments.compare(QStringLiteral("NCE_ZSCL"),Qt::CaseInsensitive) == 0)
				{
					this->Send("NCE_ZSCL      0.000010\r\n");
				}
				else if(arguments.compare(QStringLiteral("REF_MARK"),Qt::CaseInsensitive) == 0)
				{
					this->Send("REF_MARK  0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000\r\n");
				}
				else if(arguments.compare(QStringLiteral("RELAYS"),Qt::CaseInsensitive) == 0)	// SHOW RELAYS may not be a leitz command
				{
					this->Send("ENABLED\r\n");
				}
				else if(arguments.compare(QStringLiteral("REVRS"),Qt::CaseInsensitive) == 0)
				{
					this->Send("REVRS        -4,   -4,   4,   0,   0,   0\r\n");
				}
				else if(arguments.compare(QStringLiteral("SCAPAR"),Qt::CaseInsensitive) == 0)	// SHOW SCAPAR may not be a leitz command
				{
					this->Send("SCAPAR  0, 2.00000, 5.00000\r\n");
				}
				else if(arguments.compare(QStringLiteral("SCL_RES"),Qt::CaseInsensitive) == 0)
				{
					this->Send("SCL_RES  0.00100, 0.00100, 0.00100, 0.00100, 0.00100, 0.00100\r\n");
				}
				else if(arguments.compare(QStringLiteral("SCN_ACC"),Qt::CaseInsensitive) == 0)
				{
					this->Send("SCN_ACC  10.00000\r\n");
				}
				else if(arguments.compare(QStringLiteral("SCNP_OFF"),Qt::CaseInsensitive) == 0)
				{
					this->Send("SCNP_OFF  0.10000\r\n");
				}
				else if(arguments.compare(QStringLiteral("SP6_MTX"),Qt::CaseInsensitive) == 0)
				{
					this->Send("SP6MTX 0.100000,  0.00000,  0.000000,\r\n       0.000000, 0.100000, 0.000000,\r\n       0.000000, 0.000000, -0.100000\r\n%\r\n");
				}
				else if(arguments.compare(QStringLiteral("STATUS"),Qt::CaseInsensitive) == 0)
				{
					//                 System Status
					// ------------------------------------------
					// Error Status : EStop_pb          Feedhold Status : Use Jogbox Rate
					// Sloop_Status : in_estop            Fgrnd_Status  : sloop_in_estop
					// Error Status : no_error          Feedhold Status : Use Jogbox Rate
					// Sloop_Status : running             Fgrnd_Status  : in_position
					// Wrist Status : ph9_ok_manual       Autzer_Done   : TRUE
					// Probe_Type   : pr_ren              Probe_Seated  : TRUE
					
					write_text = QStringLiteral("\r\n                System Status\r\n");
					write_text += QStringLiteral("------------------------------------------\r\n");
					
					if(d_estop_status)
					{
						write_text += QStringLiteral("Error Status : EStop_pb          Feedhold Status : Use Jogbox Rate\r\n");
						write_text += QStringLiteral("Sloop_Status : in_estop            Fgrnd_Status  : sloop_in_estop\r\n");
					}
					else
					{
						write_text += QStringLiteral("Error Status : no_error          Feedhold Status : Use Jogbox Rate\r\n");
						write_text += QStringLiteral("Sloop_Status : running             Fgrnd_Status  : in_position\r\n");
					}
					
					if(d_homed_status)
					{
						write_text += QStringLiteral("Wrist Status : ph9_ok_manual       Autzer_Done   : TRUE\r\n");
						write_text += QStringLiteral("Probe_Type   : pr_ren              Probe_Seated  : TRUE\r\n");
					}
					else
					{
						write_text += QStringLiteral("Wrist Status : ph9_ok_manual       Autzer_Done   : FALSE\r\n");
						write_text += QStringLiteral("Probe_Type   : pr_ren              Probe_Seated  : TRUE\r\n");
					}
					
					this->Send(write_text.toLatin1());
				}
				else if(arguments.compare(QStringLiteral("VMAX"),Qt::CaseInsensitive) == 0)
				{
					this->Send("VMAX        500.00000,   500.00000,   500.00000,   500.00000,   500.00000\r\n");
				}
				else if(arguments.compare(QStringLiteral("VOLCOMP"),Qt::CaseInsensitive) == 0)
				{
					this->Send("VOLCOMP 1\r\n");
				}
				else if(arguments.compare(QStringLiteral("WKPCMP"),Qt::CaseInsensitive) == 0)	// SHOW WKPCMP may not be a leitz command
				{
					this->Send("DISABLED\r\n");
				}
				else if(arguments.compare(QStringLiteral("X_SENS"),Qt::CaseInsensitive) == 0)
				{
					this->Send("X_SENS      1,2,0\r\n");
				}
				else if(arguments.compare(QStringLiteral("Y_SENS"),Qt::CaseInsensitive) == 0)
				{
					this->Send("Y_SENS      3,4,0\r\n");
				}
				else if(arguments.compare(QStringLiteral("Z_SENS"),Qt::CaseInsensitive) == 0)
				{
					this->Send("Z_SENS      5,6,0\r\n");
				}
			}
			else if(keyword.compare(QStringLiteral("STATUS")) == 0)
			{
				//                 System Status
				// ------------------------------------------
				// Error Status : EStop_pb          Feedhold Status : Use Jogbox Rate
				// Sloop_Status : in_estop            Fgrnd_Status  : sloop_in_estop
				// Error Status : no_error          Feedhold Status : Use Jogbox Rate
				// Sloop_Status : running             Fgrnd_Status  : in_position
				// Wrist Status : ph9_ok_manual       Autzer_Done   : TRUE
				// Probe_Type   : pr_ren              Probe_Seated  : TRUE
				
				write_text = QStringLiteral("\r\n                System Status\r\n");
				write_text += QStringLiteral("------------------------------------------\r\n");
				
				if(d_estop_status)
				{
					write_text += QStringLiteral("Error Status : EStop_pb          Feedhold Status : Use Jogbox Rate\r\n");
					write_text += QStringLiteral("Sloop_Status : in_estop            Fgrnd_Status  : sloop_in_estop\r\n");
				}
				else
				{
					write_text += QStringLiteral("Error Status : no_error          Feedhold Status : Use Jogbox Rate\r\n");
					write_text += QStringLiteral("Sloop_Status : running             Fgrnd_Status  : in_position\r\n");
				}
				
				write_text += QStringLiteral("Wrist Status : ph9_ok_manual       Autzer_Done   : TRUE\r\n");
				write_text += QStringLiteral("Probe_Type   : pr_ren              Probe_Seated  : TRUE\r\n");
				
				this->Send(write_text.toLatin1());
			}
			else if(keyword.compare(QStringLiteral("WKPPAR")) == 0)
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("WKPTCSY")) == 0)
			{
				this->Send("%\r\n");
			}
			else if(keyword.compare(QStringLiteral("WRISTCSY")) == 0)
			{
				this->Send("%\r\n");
			}
			
#ifdef DEBUG_MODE_LEVEL_0
			else
			{
				qWarning(QString("IGND: %1").arg(line).toLatin1());
			}
#endif
		}
		
	}while(!(rx_line_index < 0));
}

QByteArray TMachineLeitz::KeyPress_Done(void) const
{
	QByteArray							data;
	
	data.append(0x8f);
	data.append(0x50);
	data.append(0x8f);
	data.append(0x4d);

	return data;
}

QByteArray TMachineLeitz::KeyPress_DelPnt(void) const
{
	QByteArray							data;

	data.append(0x8f);
	data.append(0x50);
	data.append(0x8f);
	data.append(0x53);

	return data;
}

void TMachineLeitz::Set_EStop(
	const bool 							state)
{
	d_estop_status = state;
	
	if(d_estop_status)
	{
		d_command_executor.Abort();
		
		d_jog_timer->stop();
		d_control_timer->stop();

		this->Send("ERROR MOTOR_RE,S%1,TR_FG013,E-Stop Push-Button Pressed\r\n");
	}
}

bool TMachineLeitz::Move_To(
	const TVector3						&pos)
{
	TCommandExecutor::TCommand			command;
	TVector3							target(pos);
	
	if(d_estop_status)
	{
		d_last_error = QStringLiteral("ERR:  Machine in E-Stop");
		return false;
	}

	target -= d_tool_offset;

	if(target.x < d_min_machine.x) target.x = d_min_machine.x;
	if(target.y < d_min_machine.y) target.y = d_min_machine.y;
	if(target.z < d_min_machine.z) target.z = d_min_machine.z;
	if(target.x > d_max_machine.x) target.x = d_max_machine.x;
	if(target.y > d_max_machine.y) target.y = d_max_machine.y;
	if(target.z > d_max_machine.z) target.z = d_max_machine.z;
	
	target += d_tool_offset;
	
	command.command_type = TCommandExecutor::COMMAND_MOVE;
	command.target = pos;
	
	command.speed = d_move_speed;
	
	command.completion_text = QString();
	
	d_command_executor.Add_Command(command);
	
	if(!d_jog_timer->isActive())
	{
		d_jog_timer->start(TIMER_INTERVAL);
	}
	
	return true;
}

bool TMachineLeitz::Manual_Touch(
	const TVector3						&pos,
	const TVector3						&vec)
{
	TCommandExecutor::TCommand			command;
	QString								text;
	QByteArray							bytes;
	TVector3							touch_pos(pos);
	TVector3							touch_vec(vec);
	TVector3							clear_pos;
	bool								x_error,y_error,z_error;
	static const double					ZERO_EPSILON(0.001);

	if(d_estop_status)
	{
		d_last_error = QStringLiteral("ERR:  Machine in E-Stop");
		return false;
	}

	// scale errors
	touch_pos.x *= d_scale_factor_x;
	touch_pos.y *= d_scale_factor_y;
	touch_pos.z *= d_scale_factor_z;
		
	// squareness errors
	touch_pos.x += pos.y * d_squareness_xy;
	touch_pos.y += pos.z * d_squareness_yz;
	touch_pos.x += pos.z * d_squareness_zx;

	bytes.resize(2);
	
	if(touch_vec.Length() > ZERO_EPSILON)
	{
		touch_vec.Normal();
	}
	else
	{
		d_last_error = QStringLiteral("ERR:  Touch vector not defined");
		return false;
	}
	
	text = QString("Touch %1, %2, %3, %4, %5, %6")
				.arg(touch_pos.x,0,'f',3)
				.arg(touch_pos.y,0,'f',3)
				.arg(touch_pos.z,0,'f',3)
				.arg(touch_vec.i,0,'f',5)
				.arg(touch_vec.j,0,'f',5)
				.arg(touch_vec.k,0,'f',5);
	
	emit Log_Text(text);
	
	clear_pos = touch_pos;
	
	// touch IJK is approach but machine expects surface normal
	touch_vec *= -1.0;


	text = QString("POINT %1, %2, %3, 0.00000, %4, %5, %6\r\n")
						.arg(touch_pos.x,0,'f',5)
						.arg(touch_pos.y,0,'f',5)
						.arg(touch_pos.z,0,'f',5)
						.arg(touch_vec.i,0,'f',5)
						.arg(touch_vec.j,0,'f',5)
						.arg(touch_vec.k,0,'f',5);
	
	text.insert(0,QChar(static_cast<unsigned char>(0x4f)));
	text.insert(0,QChar(static_cast<unsigned char>(0x8f)));
//	text.insert(0,(unsigned char)(0x4f));
//	text.insert(0,(unsigned char)(0x8f));	// prefix is 0x8f4f
	
	if(this->Is_Position_Inside_Volume(clear_pos,&x_error,&y_error,&z_error))
	{
		command.command_type = TCommandExecutor::COMMAND_MOVE;
		command.target = clear_pos;
		command.speed = d_jog_highspeed;
		command.completion_text = text;
		
		d_command_executor.Add_Command(command);
		
		if(!d_control_timer->isActive())
		{
			d_control_timer->start(TIMER_INTERVAL);
		}
	}
	else
	{
		text = QString();
		
		if(x_error)
		{
			text += QStringLiteral("X");
		}
		
		if(y_error)
		{
			text += QStringLiteral("Y");
		}
		
		if(z_error)
		{
			text += QStringLiteral("Z");
		}
		
		d_last_error = QString("ERR:  Axis %1 outside of machine volume.").arg(text);

		
		return false;
	}

	return true;
}

bool TMachineLeitz::Jog_Move(
	const TVector3						&direction)
{
	TCommandExecutor::TCommand			command;
	TVector3							target(d_command_executor.Current_Position());
	
	if(d_estop_status)
	{
		d_last_error = QStringLiteral("ERR:  Machine in E-Stop");
		return false;
	}
	
	if(d_control_timer->isActive())
	{
		d_last_error = QStringLiteral("ERR:  Machine currently active");
		return false;
	}
	
	target += (d_jog_distance * direction);

	target -= d_tool_offset;
	
	if(target.x < d_min_machine.x) target.x = d_min_machine.x;
	if(target.y < d_min_machine.y) target.y = d_min_machine.y;
	if(target.z < d_min_machine.z) target.z = d_min_machine.z;
	if(target.x > d_max_machine.x) target.x = d_max_machine.x;
	if(target.y > d_max_machine.y) target.y = d_max_machine.y;
	if(target.z > d_max_machine.z) target.z = d_max_machine.z;
	
	target += d_tool_offset;
	
	command.command_type = TCommandExecutor::COMMAND_MOVE;
	command.target = target;
	
	command.speed = d_move_speed;
	
	command.completion_text = QString();
	
	d_command_executor.Add_Command(command);
	
	if(!d_jog_timer->isActive())
	{
		d_jog_timer->start(TIMER_INTERVAL);
	}
	
	return true;
}

TVector3 TMachineLeitz::Get_Move_Data(
	const QString 						&text) const
{
	QStringList							list;
	TVector3							pos;
	TVector3							current_pos;
	bool								state;
	
	list = text.split(',');
	
	if(list.size() > 2)
	{
		current_pos = this->Position();
		
		pos.x = list[0].toDouble(&state);
		if(!state) pos.x = current_pos.x;
		
		pos.y = list[1].toDouble(&state);
		if(!state) pos.y = current_pos.y;
		
		pos.z = list[2].toDouble(&state);
		if(!state) pos.z = current_pos.z;
	}
	
	return pos;
}

double TMachineLeitz::Get_Move_Speed(
	const QString 						&text) const
{
	QStringList							list;
	double								speed(500);
	
	list = text.split(',');
	
	if(list.size() > 2)
	{
		speed = list[0].toDouble();
		speed += list[1].toDouble();
		speed += list[2].toDouble();
		
		speed /= 3.0;
	}
	
	return speed;
}

bool TMachineLeitz::Get_Touch_Data(
	const QString						&text,
	TVector3							* const xyz,
	TVector3							* const ijk) const
{
	QStringList							list;
	TVector3							pnt;
	TVector3							vec;
	static const double					ZERO_EPSILON(0.001);
	
	assert(xyz);
	assert(ijk);
	
	list = text.split(',');
	
	if(list.size() > 6)
	{
		pnt.x = list[0].toDouble();
		pnt.y = list[1].toDouble();
		pnt.z = list[2].toDouble();
		
		(*xyz) = pnt;
			
		vec.i = list[4].toDouble();
		vec.j = list[5].toDouble();
		vec.k = list[6].toDouble();
		
		if(vec.Length() > ZERO_EPSILON)
		{
			vec.Normal();
		}
		else
		{
			vec.Set(0,0,1);
		}
		
		*ijk = vec;
			
		// scale errors
		(*xyz).x *= d_scale_factor_x;
		(*xyz).y *= d_scale_factor_y;
		(*xyz).z *= d_scale_factor_z;

		// squareness errors
		(*xyz).x += pnt.y * d_squareness_xy;
		(*xyz).y += pnt.z * d_squareness_yz;
		(*xyz).x += pnt.z * d_squareness_zx;

		return true;
	}
	
	return false;
}

bool TMachineLeitz::Get_Head_Angles(
	const QString						&text,
	double								* const a,
	double								* const b) const
{
	QStringList							list;
	
	assert(a);
	assert(b);
	
	list = text.split(',');		// PH9, A, B
	
	if(list.size() > 2)
	{
		// list[0] is PH9
		*a = list[1].toDouble();
		*b = list[2].toDouble();
		
		return true;
	}
	
	return false;
}


void TMachineLeitz::Update_Touch_Parameters(
	const QString						&text)
{
	QStringList							list;
	
	list = text.split(',');
	
	if(list.size() > 2)
	{
		d_touch_speed = list[0].toDouble();
		d_prehit_distance = list[2].toDouble();
	}
}

void TMachineLeitz::Update_Tool_Data(
	const QString						&text)
{
	QStringList							list;
	TVector3							pos;
	
	list = text.split(',');
	
	if(list.size() > 2)
	{
		pos.x = list[0].toDouble();
		pos.y = list[1].toDouble();
		pos.z = list[2].toDouble();
		
		d_tool_offset = pos;
	}
	
	if(list.size() > 12)
	{
		d_prbpin_tip_radius = list[12].toDouble() / 2.0;
	}
	
	if(d_use_override_tip_radius)
	{
		d_active_tip_radius = d_override_tip_radius;
	}
	else
	{
		d_active_tip_radius = d_prbpin_tip_radius;
	}
}
