/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QApplication>
#include <QFrame>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QCheckBox>
#include <QSpacerItem>

#include "machineerrorsdialog.h"

TMachineErrorsDialog::TMachineErrorsDialog(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFrame								*separator_hline;
	QGridLayout							*dialog_layout;
	QHBoxLayout							*button_hlayout;
	QLabel								*scale_x_label;
	QLabel								*scale_x_units_label;
	QLabel								*scale_y_label;
	QLabel								*scale_y_units_label;
	QLabel								*scale_z_label;
	QLabel								*scale_z_units_label;
	QLabel								*squareness_xy_label;
	QLabel								*squareness_xy_units_label;
	QLabel								*squareness_yz_label;
	QLabel								*squareness_yz_units_label;
	QLabel								*squareness_zx_label;
	QLabel								*squareness_zx_units_label;
	QPushButton							*cancel_button;
	QSpacerItem							*button_hspacer;
	QSpacerItem							*dialog_vspacer;

	this->resize(363,303);

	dialog_layout = new QGridLayout(this);
	
	scale_x_label = new QLabel(this);
	dialog_layout->addWidget(scale_x_label,0,0,1,1);

	d_scale_x_edit = new QLineEdit(this);
	d_scale_x_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_scale_x_edit,0,1,1,1);

	scale_x_units_label = new QLabel(this);
	dialog_layout->addWidget(scale_x_units_label,0,2,1,1);

	scale_y_label = new QLabel(this);
	dialog_layout->addWidget(scale_y_label,1,0,1,1);

	d_scale_y_edit = new QLineEdit(this);
	d_scale_y_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_scale_y_edit,1,1,1,1);

	scale_y_units_label = new QLabel(this);
	dialog_layout->addWidget(scale_y_units_label,1,2,1,1);
	
	scale_z_label = new QLabel(this);
	dialog_layout->addWidget(scale_z_label,2,0,1,1);
	
	d_scale_z_edit = new QLineEdit(this);
	d_scale_z_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_scale_z_edit,2,1,1,1);

	scale_z_units_label = new QLabel(this);
	dialog_layout->addWidget(scale_z_units_label,2,2,1,1);

	squareness_xy_label = new QLabel(this);
	dialog_layout->addWidget(squareness_xy_label,3,0,1,1);

	d_squareness_xy_edit = new QLineEdit(this);
	d_squareness_xy_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_squareness_xy_edit,3,1,1,1);

	squareness_xy_units_label = new QLabel(this);
	dialog_layout->addWidget(squareness_xy_units_label,3,2,1,1);

	squareness_yz_label = new QLabel(this);
	dialog_layout->addWidget(squareness_yz_label,4,0,1,1);

	d_squareness_yz_edit = new QLineEdit(this);
	d_squareness_yz_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_squareness_yz_edit,4,1,1,1);

	squareness_yz_units_label = new QLabel(this);
	dialog_layout->addWidget(squareness_yz_units_label,4,2,1,1);

	squareness_zx_label = new QLabel(this);
	dialog_layout->addWidget(squareness_zx_label,5,0,1,1);
	
	d_squareness_zx_edit = new QLineEdit(this);
	d_squareness_zx_edit->setAlignment(Qt::AlignCenter);
	dialog_layout->addWidget(d_squareness_zx_edit,5,1,1,1);

	squareness_zx_units_label = new QLabel(this);
	dialog_layout->addWidget(squareness_zx_units_label,5,2,1,1);
	
	d_correct_ballbar_check = new QCheckBox(this);
	dialog_layout->addWidget(d_correct_ballbar_check,6,1,1,2);

	dialog_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	dialog_layout->addItem(dialog_vspacer,7,0,1,3);
	
	separator_hline = new QFrame(this);
	separator_hline->setFrameShape(QFrame::HLine);
	separator_hline->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(separator_hline,8,0,1,3);

	button_hlayout = new QHBoxLayout();

	button_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	button_hlayout->addItem(button_hspacer);

	cancel_button = new QPushButton(this);
	cancel_button->setAutoDefault(false);
	button_hlayout->addWidget(cancel_button);

	d_accept_button = new QPushButton(this);
	d_accept_button->setAutoDefault(false);
	button_hlayout->addWidget(d_accept_button);
	dialog_layout->addLayout(button_hlayout,9,0,1,3);

	// defaults
	d_correct_ballbar_check->setChecked(true);

	QWidget::setTabOrder(d_scale_x_edit,d_scale_y_edit);
	QWidget::setTabOrder(d_scale_y_edit,d_scale_z_edit);
	QWidget::setTabOrder(d_scale_z_edit,d_squareness_xy_edit);
	QWidget::setTabOrder(d_squareness_xy_edit,d_squareness_yz_edit);
	QWidget::setTabOrder(d_squareness_yz_edit,d_squareness_zx_edit);
	QWidget::setTabOrder(d_squareness_zx_edit,d_correct_ballbar_check);

	this->setWindowTitle(QStringLiteral("Machine Errors"));
	
	squareness_xy_label->setText(QStringLiteral("Squareness Error XY:"));
	squareness_yz_label->setText(QStringLiteral("Squareness Error YZ:"));
	squareness_zx_label->setText(QStringLiteral("Squareness Error ZX:"));
	squareness_xy_units_label->setText(QStringLiteral("mm/m"));
	squareness_yz_units_label->setText(QStringLiteral("mm/m"));
	squareness_zx_units_label->setText(QStringLiteral("mm/m"));
	scale_x_label->setText(QStringLiteral("Scale Error X:"));
	scale_y_label->setText(QStringLiteral("Scale Error Y:"));
	scale_z_label->setText(QStringLiteral("Scale Error Z:"));
	scale_x_units_label->setText(QStringLiteral("mm/m"));
	scale_y_units_label->setText(QStringLiteral("mm/m"));
	scale_z_units_label->setText(QStringLiteral("mm/m"));
	d_scale_x_edit->setText(QStringLiteral("0.0000"));
	d_scale_y_edit->setText(QStringLiteral("0.0000"));
	d_scale_z_edit->setText(QStringLiteral("0.0000"));
	d_squareness_xy_edit->setText(QStringLiteral("0.0000"));
	d_squareness_yz_edit->setText(QStringLiteral("0.0000"));
	d_squareness_zx_edit->setText(QStringLiteral("0.0000"));
	d_correct_ballbar_check->setText(QStringLiteral("Correct ballbar measurements."));
	cancel_button->setText(QStringLiteral("Cancel"));
	d_accept_button->setText(QStringLiteral("Accept"));

	connect(d_accept_button,&QPushButton::clicked,this,&TMachineErrorsDialog::accept);
	connect(cancel_button,&QPushButton::clicked,this,&TMachineErrorsDialog::reject);
}

TMachineErrorsDialog::~TMachineErrorsDialog(void)
{
}

double TMachineErrorsDialog::Scale_X(void) const
{
	return d_scale_x_edit->text().toDouble();
}

double TMachineErrorsDialog::Scale_Y(void) const
{
	return d_scale_y_edit->text().toDouble();
}

double TMachineErrorsDialog::Scale_Z(void) const
{
	return d_scale_z_edit->text().toDouble();
}

double TMachineErrorsDialog::Squareness_XY(void) const
{
	return d_squareness_xy_edit->text().toDouble();
}

double TMachineErrorsDialog::Squareness_YZ(void) const
{
	return d_squareness_yz_edit->text().toDouble();
}

double TMachineErrorsDialog::Squareness_ZX(void) const
{
	return d_squareness_zx_edit->text().toDouble();
}

bool TMachineErrorsDialog::Correct_Ballbar(void) const
{
	return d_correct_ballbar_check->isChecked();
}

void TMachineErrorsDialog::Set_Scale_X(
	const double			 			&error)
{
	d_scale_x_edit->setText(QString("%1").arg(error,0,'f',4));
}

void TMachineErrorsDialog::Set_Scale_Y(
	const double			 			&error)
{
	d_scale_y_edit->setText(QString("%1").arg(error,0,'f',4));
}

void TMachineErrorsDialog::Set_Scale_Z(
	const double			 			&error)
{
	d_scale_z_edit->setText(QString("%1").arg(error,0,'f',4));
}

void TMachineErrorsDialog::Set_Squareness_XY(
	const double			 			&error)
{
	d_squareness_xy_edit->setText(QString("%1").arg(error,0,'f',4));
}

void TMachineErrorsDialog::Set_Squareness_YZ(
	const double			 			&error)
{
	d_squareness_yz_edit->setText(QString("%1").arg(error,0,'f',4));
}

void TMachineErrorsDialog::Set_Squareness_ZX(
	const double			 			&error)
{
	d_squareness_zx_edit->setText(QString("%1").arg(error,0,'f',4));
}

void TMachineErrorsDialog::Set_Correct_Ballbar(
	const bool 							state)
{
	d_correct_ballbar_check->setChecked(state);
}

