/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QGridLayout>
#include <QLabel>
#include <QSizePolicy>
#include <QToolButton>
#include <QFrame>
#include <QSpacerItem>
#include <QLabel>
#include <QIcon>
#include <cmath>

#include "../../core/mat4.h"
#include "../../core/openglgeometryfactory.h"

#include "opengldisplay.h"

#include "opengldisplaywidget.h"

static const double						ZERO_EPSILON = 0.0000001;

TOpenGLDisplayWidget::TOpenGLDisplayWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QFrame(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*widget_layout;
	QSizePolicy							size_policy(QSizePolicy::Maximum,QSizePolicy::Preferred);
	QSpacerItem							*widget_vspacer;
	QString								gl_error_text;
	QString								gl_version_text;
	QString								gl_extensions_text;
	int									gl_major_version;
	
	// check version of OpenGL.  Display will not work if opengl version is less than 2
	gl_major_version = TOpenGLWidget::Get_OpenGL_Version_Major(&gl_version_text,&gl_extensions_text);
	
	size_policy.setHorizontalStretch(0);
	size_policy.setVerticalStretch(0);
	
	this->setFrameShape(QFrame::Box);
	this->setStyleSheet(QStringLiteral("background-color: rgb(255, 255, 255);"));
	
	widget_layout = new QGridLayout(this);
	widget_layout->setContentsMargins(1,1,1,1);

	if(gl_major_version > 1)
	{
		d_opengl_display = new TOpenGLDisplay(this);
		widget_layout->addWidget(d_opengl_display,0,0,6,1);
		
		d_gldisplay_error_label = 0;
	}
	else
	{
		d_gldisplay_error_label = new QLabel(this);
		d_gldisplay_error_label->setWordWrap(true);
		d_gldisplay_error_label->setAlignment(Qt::AlignLeading|Qt::AlignLeft|Qt::AlignTop);
		widget_layout->addWidget(d_gldisplay_error_label,0,0,6,1);
		
		d_opengl_display = 0;
	}

	d_scale_to_fit_button = new QToolButton(this);
	d_scale_to_fit_button->setMaximumSize(24,24);
	d_scale_to_fit_button->setCheckable(false);
	widget_layout->addWidget(d_scale_to_fit_button,0,1,1,1);

	d_pan_button = new QToolButton(this);
	d_pan_button->setMaximumSize(24,24);
	d_pan_button->setCheckable(true);
	widget_layout->addWidget(d_pan_button,1,1,1,1);

	d_rotate_2d_button = new QToolButton(this);
	d_rotate_2d_button->setMaximumSize(24,24);
	d_rotate_2d_button->setCheckable(true);
	widget_layout->addWidget(d_rotate_2d_button,2,1,1,1);

	d_rotate_3d_button = new QToolButton(this);
	d_rotate_3d_button->setMaximumSize(24,24);
	d_rotate_3d_button->setCheckable(true);
	widget_layout->addWidget(d_rotate_3d_button,3,1,1,1);
	
	widget_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	widget_layout->addItem(widget_vspacer,4,1,1,1);

	d_scale_to_fit_button->setText(QStringLiteral("..."));
	d_pan_button->setText(QStringLiteral("..."));
	d_rotate_2d_button->setText(QStringLiteral("..."));
	d_rotate_3d_button->setText(QStringLiteral("..."));
	
	d_scale_to_fit_button->setIcon(QIcon(":/toolbar/tb_scale_to_fit.png"));
	d_pan_button->setIcon(QIcon(":/toolbar/tb_pan.png"));
	d_rotate_2d_button->setIcon(QIcon(":/toolbar/tb_rotate_2d.png"));
	d_rotate_3d_button->setIcon(QIcon(":/toolbar/tb_rotate_3d.png"));
	
	d_scale_to_fit_button->setToolTip(QStringLiteral("Scale to Fit"));
	d_pan_button->setToolTip(QStringLiteral("Set_Pan Mode"));
	d_rotate_2d_button->setToolTip(QStringLiteral("Rotate 2D Mode"));
	d_rotate_3d_button->setToolTip(QStringLiteral("Rotate 3D Mode"));
	
	if(d_gldisplay_error_label)
	{
		gl_error_text = QStringLiteral("<html><head/><body>");
		gl_error_text += QStringLiteral("<p><span style=\" font-weight:700;\">ERR:  OpenGL version not supported.  No graphical display will be shown.</span></p>");
		gl_error_text += QStringLiteral("<p><span style=\" font-weight:500; text-decoration: underline;\">OpenGL Version Text:</span></p>");
		gl_error_text += QString("<p>%1</p>").arg(gl_version_text);
		gl_error_text += QStringLiteral("<p><span style=\" font-weight:500; text-decoration: underline;\">OpenGL Extensions:</span></p>");
		gl_error_text += QString("<p>%1</p>").arg(gl_extensions_text);
		gl_error_text += QStringLiteral("</body></html>");
		
		d_gldisplay_error_label->setText(gl_error_text);
	}
	
	// defaults
	this->Define_Machine_Volume(TVector3(-100,-100,-100),TVector3(100,100,100));
	
	d_tool_offset = TVector3(0,0,0);
	d_tip_radius = 2.0;
	d_head_angle_a = 0.0;
	d_head_angle_b = 0.0;
	this->Draw_Sensor();
	
	d_pan_button->setChecked(true);
	
	connect(d_scale_to_fit_button,SIGNAL(clicked(void)),this,SLOT(Scale_To_Fit_Clicked(void)));
	connect(d_pan_button,SIGNAL(toggled(bool)),this,SLOT(Pan_Clicked(bool)));
	connect(d_rotate_2d_button,SIGNAL(toggled(bool)),this,SLOT(Rotate_2d_Clicked(bool)));
	connect(d_rotate_3d_button,SIGNAL(toggled(bool)),this,SLOT(Rotate_3d_Clicked(bool)));
	
	if(d_opengl_display)
	{
		connect(d_opengl_display,SIGNAL(Model_Click_Point(const double&,const double&,const double&,const int)),this,SIGNAL(Model_Item_Clicked(const double&,const double&,const double&,const int)));
	}
}

TOpenGLDisplayWidget::~TOpenGLDisplayWidget(void)
{
}

void TOpenGLDisplayWidget::Define_Machine_Volume(
	const TVector3						&pnt_min,
	const TVector3						&pnt_max)
{
	double								dval;
	
	if(!d_opengl_display)
	{
		return;
	}
	
	d_max_xyz = pnt_max;
	d_min_xyz = pnt_min;
	
	if(d_min_xyz.x > d_max_xyz.x)
	{
		dval = d_max_xyz.x;
		d_max_xyz.x = d_min_xyz.x;
		d_min_xyz.x = dval;
	}
	
	if(d_min_xyz.y > d_max_xyz.y)
	{
		dval = d_max_xyz.y;
		d_max_xyz.y = d_min_xyz.y;
		d_min_xyz.y = dval;
	}
	
	if(d_min_xyz.z > d_max_xyz.z)
	{
		dval = d_max_xyz.z;
		d_max_xyz.z = d_min_xyz.z;
		d_min_xyz.z = dval;
	}

	d_opengl_display->Define_Machine_Volume(d_min_xyz,d_max_xyz);
	
	this->Draw_Machine();
}
void TOpenGLDisplayWidget::Set_Tool(
	const TVector3						&probe_offset,
	const double						&tip_diameter)
{
	d_tool_offset = probe_offset;
	d_tip_radius = tip_diameter / 2.0;

	this->Draw_Sensor();
}

void TOpenGLDisplayWidget::Set_Head_AB(
	const double						&a,
	const double						&b)
{
	d_head_angle_a = a;
	d_head_angle_b = b;
	
	this->Draw_Sensor();
}

void TOpenGLDisplayWidget::Move_To(
	const TVector3						&pos)
{
	TVector3							machine_pos;
	
	machine_pos = pos;
	machine_pos -= d_tool_offset;
		
	if(d_opengl_display)
	{
		d_opengl_display->Move_To(machine_pos);
		d_opengl_display->Path_To(pos);
		
		d_opengl_display->update();
	}
}

void TOpenGLDisplayWidget::Add_Touch(
	const TVector3						&pos)
{
	if(d_opengl_display)
	{
		d_opengl_display->Add_Touch(pos);
	}
}

void TOpenGLDisplayWidget::Set_Touch_Buffer_Size(
	const int							buffer_size)
{
	if(d_opengl_display)
	{
		d_opengl_display->Set_Touch_Buffer_Size(buffer_size);
	}
}

void TOpenGLDisplayWidget::Update(
	const bool							scale_to_fit)
{
	if(!d_opengl_display)
	{
		return;
	}
	
	d_opengl_display->update();

	if(scale_to_fit)
	{
		d_opengl_display->Scale_To_Fit();
	}
}

void TOpenGLDisplayWidget::Scale_To_Fit_Clicked(void)
{
	if(d_opengl_display)
	{
		d_opengl_display->Scale_To_Fit();
	}
}

void TOpenGLDisplayWidget::Rotate_2d_Clicked(
	bool								toggled)
{
	bool								prev_state;
	
	if(toggled)
	{
		prev_state = d_pan_button->blockSignals(true);
		d_pan_button->setChecked(false);
		d_pan_button->blockSignals(prev_state);
	
		prev_state = d_rotate_3d_button->blockSignals(true);
		d_rotate_3d_button->setChecked(false);
		d_rotate_3d_button->blockSignals(prev_state);
		
		if(d_opengl_display)
		{
			d_opengl_display->Set_Rotate_2D();
		}
	}
	else
	{
		prev_state = d_pan_button->blockSignals(true);
		d_pan_button->setChecked(true);
		d_pan_button->blockSignals(prev_state);
		
		prev_state = d_rotate_2d_button->blockSignals(true);
		d_rotate_2d_button->setChecked(false);
		d_rotate_2d_button->blockSignals(prev_state);
		
		prev_state = d_rotate_3d_button->blockSignals(true);
		d_rotate_3d_button->setChecked(false);
		d_rotate_3d_button->blockSignals(prev_state);
		
		if(d_opengl_display)
		{
			d_opengl_display->Set_Pan();
		}
	}
}

void TOpenGLDisplayWidget::Rotate_3d_Clicked(
	bool								toggled)
{
	bool								prev_state;
	
	if(toggled)
	{
		prev_state = d_pan_button->blockSignals(true);
		d_pan_button->setChecked(false);
		d_pan_button->blockSignals(prev_state);
		
		prev_state = d_rotate_2d_button->blockSignals(true);
		d_rotate_2d_button->setChecked(false);
		d_rotate_2d_button->blockSignals(prev_state);
		
		if(d_opengl_display)
		{
			d_opengl_display->Set_Rotate_3D();
		}
	}
	else
	{
		prev_state = d_pan_button->blockSignals(true);
		d_pan_button->setChecked(true);
		d_pan_button->blockSignals(prev_state);
		
		prev_state = d_rotate_2d_button->blockSignals(true);
		d_rotate_2d_button->setChecked(false);
		d_rotate_2d_button->blockSignals(prev_state);
		
		prev_state = d_rotate_3d_button->blockSignals(true);
		d_rotate_3d_button->setChecked(false);
		d_rotate_3d_button->blockSignals(prev_state);
		
		if(d_opengl_display)
		{
			d_opengl_display->Set_Pan();
		}
	}
}

void TOpenGLDisplayWidget::Pan_Clicked(
	bool								toggled)
{
	bool								prev_state;
	
	if(toggled)
	{
		prev_state = d_rotate_3d_button->blockSignals(true);
		d_rotate_3d_button->setChecked(false);
		d_rotate_3d_button->blockSignals(prev_state);
		
		prev_state = d_rotate_2d_button->blockSignals(true);
		d_rotate_2d_button->setChecked(false);
		d_rotate_2d_button->blockSignals(prev_state);
		
		if(d_opengl_display)
		{
			d_opengl_display->Set_Pan();
		}
	}
	else
	{
		prev_state = d_pan_button->blockSignals(true);
		d_pan_button->setChecked(true);
		d_pan_button->blockSignals(prev_state);
		
		prev_state = d_rotate_2d_button->blockSignals(true);
		d_rotate_2d_button->setChecked(false);
		d_rotate_2d_button->blockSignals(prev_state);
		
		prev_state = d_rotate_3d_button->blockSignals(true);
		d_rotate_3d_button->setChecked(false);
		d_rotate_3d_button->blockSignals(prev_state);
		
		if(d_opengl_display)
		{
			d_opengl_display->Set_Pan();
		}
	}
}

void TOpenGLDisplayWidget::Draw_Machine(void)
{
	TOpenGLDisplay::TMachine			machine;
	TOpenGLGeometryFactory				geometry;
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> triangles;
	TVector3							pt[4];
	double								length;
	double								x_left,x_right,y_front,y_back;
	static const double					Z_AXIS_RADIUS(50);
	static const double					ZX_BOX_RADIUS(130);
	static const double					ZX_BOX_HEIGHT(ZX_BOX_RADIUS * 2);
	static const double					ZX_BOX_OFFSET_Z(0.5);
	static const double					X_BOX_WIDTH_HEIGHT(ZX_BOX_HEIGHT);
	static const double					X_BOX_OFFSET_Z(1);
	static const double					X_LEFT_INSIDE(50);
	static const double					X_LEFT_OUTSIDE(20);
	static const double					X_RIGHT_INSIDE(250);
	static const double					X_RIGHT_OUTSIDE(20);
	static const double					Z_TABLE_CLEARANCE(200);	
	static const double					Z_TABLE_HEIGHT(400);
		
	TOpenGLGeometryFactory::TOpenGLTriangleVertex vertex;
	
	if(!d_opengl_display)
	{
		return;
	}
	
	geometry.Set_Geometry_Resolution(1);
	
	// draw Z axis
	length = d_max_xyz.z - d_min_xyz.z;
	machine.machine_z = geometry.RenderPolygon(TVector3(0,0,length/2.0),TVector3(0,0,1),TVector3(0.707,0.707,0),Z_AXIS_RADIUS,length,4);
	
	// draw xz cover
	// +X
	pt[0].Set(ZX_BOX_RADIUS,-ZX_BOX_RADIUS,ZX_BOX_OFFSET_Z);
	pt[1].Set(ZX_BOX_RADIUS,ZX_BOX_RADIUS,ZX_BOX_OFFSET_Z);
	pt[2].Set(ZX_BOX_RADIUS,ZX_BOX_RADIUS,ZX_BOX_HEIGHT + length);
	pt[3].Set(ZX_BOX_RADIUS,-ZX_BOX_RADIUS,ZX_BOX_HEIGHT + length);
	machine.machine_xz = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	
	// +Y
	pt[0].Set(ZX_BOX_RADIUS,ZX_BOX_RADIUS,ZX_BOX_OFFSET_Z);
	pt[1].Set(-ZX_BOX_RADIUS,ZX_BOX_RADIUS,ZX_BOX_OFFSET_Z);
	pt[2].Set(-ZX_BOX_RADIUS,ZX_BOX_RADIUS,ZX_BOX_HEIGHT + length);
	pt[3].Set(ZX_BOX_RADIUS,ZX_BOX_RADIUS,ZX_BOX_HEIGHT + length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xz.insert(machine.machine_xz.end(),triangles.begin(),triangles.end());
	
	// -X
	pt[0].Set(-ZX_BOX_RADIUS,ZX_BOX_RADIUS,ZX_BOX_OFFSET_Z);
	pt[1].Set(-ZX_BOX_RADIUS,-ZX_BOX_RADIUS,ZX_BOX_OFFSET_Z);
	pt[2].Set(-ZX_BOX_RADIUS,-ZX_BOX_RADIUS,ZX_BOX_HEIGHT + length);
	pt[3].Set(-ZX_BOX_RADIUS,ZX_BOX_RADIUS,ZX_BOX_HEIGHT + length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xz.insert(machine.machine_xz.end(),triangles.begin(),triangles.end());
	
	// -Y
	pt[0].Set(-ZX_BOX_RADIUS,-ZX_BOX_RADIUS,ZX_BOX_OFFSET_Z);
	pt[1].Set(ZX_BOX_RADIUS,-ZX_BOX_RADIUS,ZX_BOX_OFFSET_Z);
	pt[2].Set(ZX_BOX_RADIUS,-ZX_BOX_RADIUS,ZX_BOX_HEIGHT + length);
	pt[3].Set(-ZX_BOX_RADIUS,-ZX_BOX_RADIUS,ZX_BOX_HEIGHT + length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xz.insert(machine.machine_xz.end(),triangles.begin(),triangles.end());

	// -Z
	pt[0].Set(-ZX_BOX_RADIUS,-ZX_BOX_RADIUS,ZX_BOX_OFFSET_Z);
	pt[1].Set(-ZX_BOX_RADIUS,ZX_BOX_RADIUS,ZX_BOX_OFFSET_Z);
	pt[2].Set(ZX_BOX_RADIUS,ZX_BOX_RADIUS,ZX_BOX_OFFSET_Z);
	pt[3].Set(ZX_BOX_RADIUS,-ZX_BOX_RADIUS,ZX_BOX_OFFSET_Z);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xz.insert(machine.machine_xz.end(),triangles.begin(),triangles.end());	

	// +Z
	pt[0].Set(-ZX_BOX_RADIUS,-ZX_BOX_RADIUS,ZX_BOX_HEIGHT + length);
	pt[1].Set(ZX_BOX_RADIUS,-ZX_BOX_RADIUS,ZX_BOX_HEIGHT + length);
	pt[2].Set(ZX_BOX_RADIUS,ZX_BOX_RADIUS,ZX_BOX_HEIGHT + length);
	pt[3].Set(-ZX_BOX_RADIUS,ZX_BOX_RADIUS,ZX_BOX_HEIGHT + length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xz.insert(machine.machine_xz.end(),triangles.begin(),triangles.end());
	
	// draw x axis
	x_left = d_min_xyz.x - ZX_BOX_RADIUS;
	x_right = d_max_xyz.x + ZX_BOX_RADIUS;

	// -Y
	pt[0].Set(x_left,0,X_BOX_OFFSET_Z);
	pt[1].Set(x_right,0,X_BOX_OFFSET_Z);
	pt[2].Set(x_right,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[3].Set(x_left,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	machine.machine_x = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	
	// +Z
	pt[0].Set(x_left,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_right,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[2].Set(x_right,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[3].Set(x_left,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_x.insert(machine.machine_x.end(),triangles.begin(),triangles.end());
	
	// +Y
	pt[0].Set(x_left,X_BOX_WIDTH_HEIGHT,X_BOX_OFFSET_Z);
	pt[1].Set(x_left,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[2].Set(x_right,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[3].Set(x_right,X_BOX_WIDTH_HEIGHT,X_BOX_OFFSET_Z);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_x.insert(machine.machine_x.end(),triangles.begin(),triangles.end());	
	
	// -Z
	pt[0].Set(x_left,0,X_BOX_OFFSET_Z);
	pt[1].Set(x_left,X_BOX_WIDTH_HEIGHT,X_BOX_OFFSET_Z);
	pt[2].Set(x_right,X_BOX_WIDTH_HEIGHT,X_BOX_OFFSET_Z);
	pt[3].Set(x_right,0,X_BOX_OFFSET_Z);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_x.insert(machine.machine_x.end(),triangles.begin(),triangles.end());	

	// draw left leg
	length = d_max_xyz.z - d_min_xyz.z;
	length += Z_TABLE_CLEARANCE;
	length += X_BOX_OFFSET_Z;
	
	x_left -= X_LEFT_INSIDE;
	x_right += X_RIGHT_INSIDE;

	// -Y
	pt[0].Set(x_left,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_left,0,-length);
	pt[2].Set(x_left + X_LEFT_INSIDE,0,-length);
	pt[3].Set(x_left + X_LEFT_INSIDE,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_x.insert(machine.machine_x.end(),triangles.begin(),triangles.end());	

	// +X
	pt[0].Set(x_left + X_LEFT_INSIDE,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_left + X_LEFT_INSIDE,0,-length);
	pt[2].Set(x_left + X_LEFT_INSIDE,X_BOX_WIDTH_HEIGHT,-length);
	pt[3].Set(x_left + X_LEFT_INSIDE,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_x.insert(machine.machine_x.end(),triangles.begin(),triangles.end());	

	// +Y
	pt[0].Set(x_left,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_left + X_LEFT_INSIDE,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[2].Set(x_left + X_LEFT_INSIDE,X_BOX_WIDTH_HEIGHT,-length);
	pt[3].Set(x_left,X_BOX_WIDTH_HEIGHT,-length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_x.insert(machine.machine_x.end(),triangles.begin(),triangles.end());	

	// +Z
	pt[0].Set(x_left,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_left + X_LEFT_INSIDE,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[2].Set(x_left + X_LEFT_INSIDE,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[3].Set(x_left,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_x.insert(machine.machine_x.end(),triangles.begin(),triangles.end());
	

	// draw right leg
	// -Y
	pt[0].Set(x_right - X_RIGHT_INSIDE,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_right - X_RIGHT_INSIDE,0,-length);
	pt[2].Set(x_right,0,-length);
	pt[3].Set(x_right,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_x.insert(machine.machine_x.end(),triangles.begin(),triangles.end());	

	// +Y
	pt[0].Set(x_right - X_RIGHT_INSIDE,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_right,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[2].Set(x_right,X_BOX_WIDTH_HEIGHT,-length);
	pt[3].Set(x_right - X_RIGHT_INSIDE,X_BOX_WIDTH_HEIGHT,-length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_x.insert(machine.machine_x.end(),triangles.begin(),triangles.end());	

	// -X
	pt[0].Set(x_right - X_RIGHT_INSIDE,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_right - X_RIGHT_INSIDE,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[2].Set(x_right - X_RIGHT_INSIDE,X_BOX_WIDTH_HEIGHT,-length);
	pt[3].Set(x_right - X_RIGHT_INSIDE,0,-length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_x.insert(machine.machine_x.end(),triangles.begin(),triangles.end());
	
	// +Z
	pt[0].Set(x_right - X_RIGHT_INSIDE,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_right,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[2].Set(x_right,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[3].Set(x_right - X_RIGHT_INSIDE,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_x.insert(machine.machine_x.end(),triangles.begin(),triangles.end());	

	// draw left leg panel
	length = d_max_xyz.z - d_min_xyz.z;
	length += Z_TABLE_CLEARANCE;
//	length += Z_TABLE_HEIGHT;

	// -Y
	pt[0].Set(x_left-X_LEFT_OUTSIDE,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_left-X_LEFT_OUTSIDE,0,-length);
	pt[2].Set(x_left,0,-length);
	pt[3].Set(x_left,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	machine.machine_xw = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);

	// +Y
	pt[0].Set(x_left-X_LEFT_OUTSIDE,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_left,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[2].Set(x_left,X_BOX_WIDTH_HEIGHT,-length);
	pt[3].Set(x_left-X_LEFT_OUTSIDE,X_BOX_WIDTH_HEIGHT,-length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xw.insert(machine.machine_xw.end(),triangles.begin(),triangles.end());	

	// -X
	pt[0].Set(x_left-X_LEFT_OUTSIDE,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_left-X_LEFT_OUTSIDE,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[2].Set(x_left-X_LEFT_OUTSIDE,X_BOX_WIDTH_HEIGHT,-length);
	pt[3].Set(x_left-X_LEFT_OUTSIDE,0,-length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xw.insert(machine.machine_xw.end(),triangles.begin(),triangles.end());
	
	// +Z
	pt[0].Set(x_left - X_LEFT_OUTSIDE,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_left,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[2].Set(x_left,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[3].Set(x_left - X_LEFT_OUTSIDE,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xw.insert(machine.machine_xw.end(),triangles.begin(),triangles.end());
	
	// -Z
	pt[0].Set(x_left,0,-length);
	pt[1].Set(x_left - X_LEFT_OUTSIDE,0,-length);
	pt[2].Set(x_left - X_LEFT_OUTSIDE,X_BOX_WIDTH_HEIGHT,-length);
	pt[3].Set(x_left,X_BOX_WIDTH_HEIGHT,-length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xw.insert(machine.machine_xw.end(),triangles.begin(),triangles.end());


	// draw right leg panel
	// -Y
	pt[0].Set(x_right,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_right,0,-length);
	pt[2].Set(x_right + X_RIGHT_OUTSIDE,0,-length);
	pt[3].Set(x_right + X_RIGHT_OUTSIDE,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xw.insert(machine.machine_xw.end(),triangles.begin(),triangles.end());	

	// +X
	pt[0].Set(x_right + X_RIGHT_OUTSIDE,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_right + X_RIGHT_OUTSIDE,0,-length);
	pt[2].Set(x_right + X_RIGHT_OUTSIDE,X_BOX_WIDTH_HEIGHT,-length);
	pt[3].Set(x_right + X_RIGHT_OUTSIDE,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xw.insert(machine.machine_xw.end(),triangles.begin(),triangles.end());	

	// +Y
	pt[0].Set(x_right,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_right + X_RIGHT_OUTSIDE,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[2].Set(x_right + X_RIGHT_OUTSIDE,X_BOX_WIDTH_HEIGHT,-length);
	pt[3].Set(x_right,X_BOX_WIDTH_HEIGHT,-length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xw.insert(machine.machine_xw.end(),triangles.begin(),triangles.end());	
	
	// +Z
	pt[0].Set(x_right,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[1].Set(x_right+X_RIGHT_OUTSIDE,0,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[2].Set(x_right+X_RIGHT_OUTSIDE,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	pt[3].Set(x_right,X_BOX_WIDTH_HEIGHT,X_BOX_WIDTH_HEIGHT + X_BOX_OFFSET_Z);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xw.insert(machine.machine_xw.end(),triangles.begin(),triangles.end());
	
	// -Z
	pt[0].Set(x_right+X_RIGHT_OUTSIDE,0,-length);
	pt[1].Set(x_right,0,-length);
	pt[2].Set(x_right,X_BOX_WIDTH_HEIGHT,-length);
	pt[3].Set(x_right+X_RIGHT_OUTSIDE,X_BOX_WIDTH_HEIGHT,-length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_xw.insert(machine.machine_xw.end(),triangles.begin(),triangles.end());

	// table
	y_front = d_min_xyz.y;
	y_back = d_max_xyz.y + ZX_BOX_HEIGHT;
	length = d_max_xyz.z - d_min_xyz.z;
	length += Z_TABLE_CLEARANCE;

	// +Z
	pt[0].Set(x_left,y_front,-length);
	pt[1].Set(x_right,y_front,-length);
	pt[2].Set(x_right,y_back,-length);
	pt[3].Set(x_left,y_back,-length);
	machine.machine_table = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);

	// -Z
	pt[0].Set(x_left,y_front,-length - Z_TABLE_HEIGHT);
	pt[1].Set(x_left,y_back,-length - Z_TABLE_HEIGHT);
	pt[2].Set(x_right,y_back,-length - Z_TABLE_HEIGHT);
	pt[3].Set(x_right,y_front,-length - Z_TABLE_HEIGHT);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_table.insert(machine.machine_table.end(),triangles.begin(),triangles.end());
	
	// +X
	pt[0].Set(x_right,y_front,-length);
	pt[1].Set(x_right,y_front,-length - Z_TABLE_HEIGHT);
	pt[2].Set(x_right,y_back,-length - Z_TABLE_HEIGHT);
	pt[3].Set(x_right,y_back,-length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_table.insert(machine.machine_table.end(),triangles.begin(),triangles.end());
	
	// +Y
	pt[0].Set(x_right,y_back,-length);
	pt[1].Set(x_right,y_back,-length - Z_TABLE_HEIGHT);
	pt[2].Set(x_left,y_back,-length - Z_TABLE_HEIGHT);
	pt[3].Set(x_left,y_back,-length);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_table.insert(machine.machine_table.end(),triangles.begin(),triangles.end());
	
	// -X
	pt[0].Set(x_left,y_front,-length - Z_TABLE_HEIGHT);
	pt[1].Set(x_left,y_front,-length);
	pt[2].Set(x_left,y_back,-length);
	pt[3].Set(x_left,y_back,-length - Z_TABLE_HEIGHT);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_table.insert(machine.machine_table.end(),triangles.begin(),triangles.end());
	
	// -Y
	pt[0].Set(x_right,y_front,-length - Z_TABLE_HEIGHT);
	pt[1].Set(x_right,y_front,-length);
	pt[2].Set(x_left,y_front,-length);
	pt[3].Set(x_left,y_front,-length - Z_TABLE_HEIGHT);
	triangles = geometry.RenderPlane(pt[0],pt[1],pt[2],pt[3]);
	machine.machine_table.insert(machine.machine_table.end(),triangles.begin(),triangles.end());

	d_opengl_display->Set_Machine(machine);
	d_opengl_display->update();
}

void TOpenGLDisplayWidget::Draw_Sensor(void)
{
	TOpenGLDisplay::TSensor				probe;
	TOpenGLGeometryFactory				geometry;
	std::vector<TOpenGLGeometryFactory::TOpenGLTriangleVertex> triangles;
	TVector3							p1,p2,p3,p4,p5;
	TVector3							axis,xy_axis;
	double								length,length_xy,length_z;
	double								sina;
	double								tip_radius(d_tip_radius);
	double								stem_radius;
	static const double					MINIMUM_TIP_RADIUS(0.5);
	static const double					PROBE_HEAD_RADIUS(30);
	static const double					PROBE_SHAFT_RADIUS(43);
	static const double					MIN_XY_PROBE_OFFSET_FOR_JOINT(20);
	static const double					MIN_A_ANGLE(2.5);	// degrees
	static const double					DEG2RAD(0.01745329251994);
	
	if(!d_opengl_display)
	{
		return;
	}
	
	if(tip_radius < MINIMUM_TIP_RADIUS)
	{
		tip_radius = MINIMUM_TIP_RADIUS;
	}
	
	stem_radius = tip_radius * 0.75;
	
	geometry.Set_Geometry_Resolution(3);
	
	// 2d offsets
	p1 = p2 = d_tool_offset;
	p1.z = 0;
	p2.x = 0;
	p2.y = 0;
	
	length_xy = p1.Length();
	length_z = p2.Length();
	
	if(length_xy > MIN_XY_PROBE_OFFSET_FOR_JOINT &&
	   d_head_angle_a < MIN_A_ANGLE)
	{
		// draw a stylus with a knuckle
		xy_axis = p1;
		xy_axis.Normal();
		
		p1.Set(0,0,0);
		p3.Set(0,0,d_tool_offset.z);
		p5 = d_tool_offset;
		
		p2 = (p1 + p3) / 2.0;
		p4 = (p3 + p5) / 2.0;
		
		probe.tool_stem = geometry.RenderCylinder(p2,TVector3(0,0,1),stem_radius,length_z);
		
		triangles = geometry.RenderPolygon(p3,TVector3(0,0,1),xy_axis,5,5,8);
		probe.tool_stem.insert(probe.tool_stem.end(),triangles.begin(),triangles.end());

		triangles = geometry.RenderCylinder(p4,xy_axis,1.5,length_xy);
		probe.tool_stem.insert(probe.tool_stem.end(),triangles.begin(),triangles.end());
	
		probe.tool_ruby = geometry.RenderSphere(p5,tip_radius);
	}
	else if(length_xy > MIN_XY_PROBE_OFFSET_FOR_JOINT &&
	   d_head_angle_a > MIN_A_ANGLE)
	{
		// draw something that looks like an indexable probe head
		
		xy_axis = p1;
		xy_axis.Normal();
		
		sina = sin(d_head_angle_a * DEG2RAD);
		
		length = length_xy / sina;
		
		axis.x = xy_axis.x * sina;
		axis.y = xy_axis.y * sina;
		axis.z = -cos(d_head_angle_a * DEG2RAD);
		
		p1.Set(0,0,0);
		p5 = d_tool_offset;
		p3 = p5 - axis * length;
		
		p2 = p3 - p1;

		p2 = (p1 + p3) / 2.0;
		p4 = (p3 + p5) / 2.0;
		
		probe.probe_head = geometry.RenderPolygon(p2,TVector3(0,0,1),TVector3(1,1,0),PROBE_SHAFT_RADIUS,(p3 - p1).Length(),4);
		
		triangles = geometry.RenderSphere(p3,PROBE_HEAD_RADIUS);
		probe.probe_head.insert(probe.probe_head.end(),triangles.begin(),triangles.end());

		probe.tool_stem = geometry.RenderCylinder(p4,axis,stem_radius,length);
		
		probe.tool_ruby = geometry.RenderSphere(p5,tip_radius);
	}
	else if(length_z > ZERO_EPSILON)
	{
		// draw something that looks like an indexable probe head at A0B0
				
		p1.Set(0,0,0);
		p5 = d_tool_offset;
		p3 = (p1 + p5) / 2.0;
		
		p2 = (p1 + p3) / 2.0;
		p4 = (p3 + p5) / 2.0;
		
		probe.probe_head = geometry.RenderPolygon(p2,TVector3(0,0,1),TVector3(1,1,0),PROBE_SHAFT_RADIUS,(p3 - p1).Length(),4);
		
		triangles = geometry.RenderSphere(p3,PROBE_HEAD_RADIUS);
		probe.probe_head.insert(probe.probe_head.end(),triangles.begin(),triangles.end());

		probe.tool_stem = geometry.RenderCylinder(p4,TVector3(0,0,1),stem_radius,(p5 - p3).Length());
		
		probe.tool_ruby = geometry.RenderSphere(p5,tip_radius);
	}

	d_opengl_display->Set_Tool(probe);
	
	d_opengl_display->update();
}
