/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2026  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////


#include <QAction>
#include <QApplication>
#include <QGridLayout>
#include <QWidget>
#include <QLabel>
#include <QFrame>
#include <QPlainTextEdit>
#include <QTabWidget>
#include <QSplitter>
#include <QDockWidget>
#include <QMenu>
#include <QMenuBar>
#include <QSpacerItem>
#include <QToolButton>
#include <QScreen>
#include <QSettings>
#include <QTcpServer>
#include <QTcpSocket>
#include <QSerialPort>
#include <QHostAddress>
#include <QTimer>
#include <QPixmap>
#include <QSizePolicy>
#include <QFont>
#include <QDir>
#include <QList>
#include <QRect>
#include <QSize>
#include <QPoint>
#include <QThread>

#include "../../core/messagebox.h"

//#include "touchpoint_dock.h"
#include "artifactcontainerwidget.h"
#include "machine_dock.h"
#include "bufferdisplaywidget.h"
#include "opengldisplaywidget.h"
#include "jogboxdialog.h"
#include "machineerrorsdialog.h"
#include "coordinatedisplaywidget.h"
#include "machine.h"
#include "machine_leitz.h"
#include "machine_dc.h"
#include "machine_ipp.h"
#include "machine_offline.h"

#include "virtualcmm.h"

TVirtualCMM::TVirtualCMM(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QMainWindow(const_cast<QWidget*>(parent),flags)
{
	QTabWidget							*display_tab;
	QDockWidget							*touchpoint_dock;
	QFrame								*separator_hline;
	QGridLayout							*grid_layout;
	QGridLayout							*information_layout;
	QLabel								*logo_label;
	QLabel								*copyright_label;
	QMenu								*file_menu;
	QMenu								*view_menu;
	QMenu								*help_menu;
	QMenuBar							*menubar;
	QWidget								*central_widget;
	QWidget								*information_widget;
	QSpacerItem							*display_vspacer;
	QSizePolicy							sizepolicy_preferred_maximum(QSizePolicy::Preferred, QSizePolicy::Maximum);
	QRect								desktop_rect;
	QRect								window_rect;
	QPoint								position;
	QSize								size;
	QDir								data_path;
	QList<int>							splitter_sizes;
	TVector3							min_machine;
	TVector3							max_machine;
	TVector3							move_position;
	TVector3							pos;
	TVector3							vec;
	double								dval;
	QFont								log_font(QStringLiteral("courier new"));
	QFont								copyright_font;
	QString								text;

	copyright_font.setFamily(QStringLiteral("Helvetica"));
	copyright_font.setPointSize(8);
	copyright_font.setItalic(true);
	
	sizepolicy_preferred_maximum.setHorizontalStretch(0);
	sizepolicy_preferred_maximum.setVerticalStretch(0);

	d_application_path = qApp->applicationDirPath();
	log_font.setFixedPitch(true);

#ifdef Q_OS_MAC
	int index = d_application_path.lastIndexOf(QStringLiteral(".app/"));
	d_application_path.truncate(index);
	
	index = d_application_path.lastIndexOf('/');
	d_application_path.truncate(index);
#endif
	
#ifdef Q_OS_WIN
	d_data_path = d_application_path;
#else
	d_data_path = QDir::homePath();
#endif

	data_path.setPath(d_data_path);
	
	if(!data_path.exists(QStringLiteral(".virtualcmm")))
	{
		data_path.mkdir(QStringLiteral(".virtualcmm"));
	}
	
	data_path.cd(QStringLiteral(".virtualcmm"));
	d_data_path = data_path.absolutePath();
	
	d_settings = new QSettings(d_data_path + QStringLiteral("/virtualcmm_settings.ini"),QSettings::IniFormat,this);
	
	position = d_settings->value(QStringLiteral("Mainwindow_Position"), QPoint(30,30)).toPoint();
	size = d_settings->value(QStringLiteral("Mainwindow_Size"), QSize(1001, 801)).toSize();
	
	desktop_rect = QGuiApplication::primaryScreen()->availableGeometry();
	
	window_rect.moveTopLeft(position);
	window_rect.setSize(size);
	
	window_rect = window_rect & desktop_rect;
	
	if(window_rect.isValid() && window_rect.width() > 500 && window_rect.height() > 500)
	{
		this->resize(window_rect.size());
		this->move(window_rect.topLeft());
	}
	else
	{
		this->resize(1001, 801);
		this->move(QPoint(30,30));
	}
	
	splitter_sizes.push_back(d_settings->value(QStringLiteral("Graphic_Window_Height"), 800).toInt());
	splitter_sizes.push_back(d_settings->value(QStringLiteral("Information_Window_Height"), 200).toInt());

	d_file_quit_action = new QAction(this);
	d_file_quit_action->setShortcut(QKeySequence::Quit);

	d_help_about_action = new QAction(this);
	
	central_widget = new QWidget(this);
	this->setCentralWidget(central_widget);
	
	grid_layout = new QGridLayout(central_widget);

	d_splitter_widget = new QSplitter(Qt::Vertical,central_widget);
	grid_layout->addWidget(d_splitter_widget);

	d_opengl_display_widget = new TOpenGLDisplayWidget();
	d_splitter_widget->addWidget(d_opengl_display_widget);
	
	information_widget = new QWidget();
	information_layout = new QGridLayout(information_widget);
	information_layout->setContentsMargins(0,0,0,0);
	
	display_tab = new QTabWidget(information_widget);
	information_layout->addWidget(display_tab,0,0,4,1);

	d_message_log = new QPlainTextEdit();
	d_message_log->setFont(log_font);
	d_message_log->setWordWrapMode(QTextOption::NoWrap);
	
	display_tab->addTab(d_message_log,QStringLiteral("Message Log"));
	
	d_communication_buffer_widget = new TBufferDisplayWidget();
	display_tab->addTab(d_communication_buffer_widget,QStringLiteral("Communication"));
	
	d_coordinate_display_widget = new TCoordinateDisplayWidget(information_widget);
	information_layout->addWidget(d_coordinate_display_widget,0,1,1,2);

	display_vspacer = new QSpacerItem(0, 0, QSizePolicy::Minimum, QSizePolicy::Expanding);
	information_layout->addItem(display_vspacer, 1, 1, 1, 2);
	
	separator_hline = new QFrame(information_widget);
	separator_hline->setFrameShape(QFrame::HLine);
	separator_hline->setFrameShadow(QFrame::Sunken);
	information_layout->addWidget(separator_hline, 2, 1, 1, 2);

	logo_label = new QLabel(information_widget);
	logo_label->setMinimumSize(100,40);
	logo_label->setMaximumSize(100,40);
	information_layout->addWidget(logo_label,3,2,1,1);
	
	copyright_label = new QLabel(information_widget);
	copyright_label->setFont(copyright_font);
	sizepolicy_preferred_maximum.setHeightForWidth(copyright_label->sizePolicy().hasHeightForWidth());
	copyright_label->setSizePolicy(sizepolicy_preferred_maximum);
	copyright_label->setAlignment(Qt::AlignBottom|Qt::AlignLeft);	// Qt::AlignRight|Qt::AlignTrailing
	information_layout->addWidget(copyright_label,4,0,1,3);

	d_splitter_widget->addWidget(information_widget);

	menubar = new QMenuBar(this);
	this->setMenuBar(menubar);

	file_menu = new QMenu(menubar);
	file_menu->addSeparator();
	file_menu->addAction(d_file_quit_action);
	menubar->addMenu(file_menu);
	
	view_menu = new QMenu(menubar);
	menubar->addMenu(view_menu);

	help_menu = new QMenu(menubar);
	help_menu->addAction(d_help_about_action);
	menubar->resize(1068,22);
	menubar->addMenu(help_menu);

	menubar->addAction(file_menu->menuAction());
	menubar->addAction(view_menu->menuAction());
	menubar->addAction(help_menu->menuAction());
	
	touchpoint_dock = new QDockWidget(this);
	
	d_artifact_container_widget = new TArtifactContainerWidget();
	touchpoint_dock->setWidget(d_artifact_container_widget);

	d_machine_dock = new TMachineDock(this);

	this->addDockWidget(static_cast<Qt::DockWidgetArea>(1),touchpoint_dock);
	this->addDockWidget(static_cast<Qt::DockWidgetArea>(1),d_machine_dock->Dock_Widget());
	
	view_menu->addAction(touchpoint_dock->toggleViewAction());
	view_menu->addAction(d_machine_dock->Dock_Widget()->toggleViewAction());
	
	// defaults
	d_splitter_widget->setSizes(splitter_sizes);
	d_tcp_socket = 0;
	d_msg_box = new TMessageBox(this);
	d_tcp_server = new QTcpServer(this);
	d_jogbox_dialog = new TJogboxDialog(this,Qt::Dialog | Qt::WindowStaysOnTopHint);
	d_protocol_dialog = new TProtocolDialog(this);
	d_serial_port = new QSerialPort(this);
	d_machine_errors_dialog = new TMachineErrorsDialog(this);
	
	text = QStringLiteral("/dev/ttyUSB0");
#ifdef Q_OS_MAC
	text = QStringLiteral("/dev/ttys001");
#endif
#ifdef Q_OS_WIN
	text = QStringLiteral("COM1");
#endif
	
	d_protocol_dialog->Set_Controller_Type(static_cast<TProtocolDialog::TControllerType>(d_settings->value(QStringLiteral("Controller_Type"), 0).toInt()));
	d_protocol_dialog->Set_Connection_Type(static_cast<TProtocolDialog::TProtocolType>(d_settings->value(QStringLiteral("Connection_Type"), 0).toInt()));
	d_protocol_dialog->Set_Comm_Port(d_settings->value(QStringLiteral("Connection_Serial_Name"), text).toString());
	d_protocol_dialog->Set_Baud_Rate(static_cast<QSerialPort::BaudRate>(d_settings->value(QStringLiteral("Connection_Baud_Rate"), 9600).toInt()));
	d_protocol_dialog->Set_Ethernet_Address(d_settings->value(QStringLiteral("Connection_Ethernet_Address"), QStringLiteral("127.0.0.1")).toString());
	d_protocol_dialog->Set_Ethernet_Port(d_settings->value(QStringLiteral("Connection_Ethernet_Port"), 2001).toInt());
	
	min_machine.x = d_settings->value(QStringLiteral("Machine_Min_X"), 0.0).toDouble();
	min_machine.y = d_settings->value(QStringLiteral("Machine_Min_Y"), 0.0).toDouble();
	min_machine.z = d_settings->value(QStringLiteral("Machine_Min_Z"), -1000.0).toDouble();
	
	max_machine.x = d_settings->value(QStringLiteral("Machine_Max_X"), 1200.0).toDouble();
	max_machine.y = d_settings->value(QStringLiteral("Machine_Max_Y"), 2200.0).toDouble();
	max_machine.z = d_settings->value(QStringLiteral("Machine_Max_Z"), 0.0).toDouble();

	move_position.x = d_settings->value(QStringLiteral("Move_Position_X"), 100.0).toDouble();
	move_position.y = d_settings->value(QStringLiteral("Move_Position_Y"), 200.0).toDouble();
	move_position.z = d_settings->value(QStringLiteral("Move_Position_Z"), -300.0).toDouble();

	if(min_machine.x > max_machine.x)
	{
		dval = min_machine.x;
		min_machine.x = max_machine.x;
		max_machine.x = dval;
	}
	
	if(min_machine.y > max_machine.y)
	{
		dval = min_machine.y;
		min_machine.y = max_machine.y;
		max_machine.y = dval;
	}
	
	if(min_machine.z > max_machine.z)
	{
		dval = min_machine.z;
		min_machine.z = max_machine.z;
		max_machine.z = dval;
	}
	
	d_machine_dock->Set_Min_Machine(min_machine);
	d_machine_dock->Set_Max_Machine(max_machine);
	d_machine_dock->Set_Move_Position(move_position);
	d_machine_dock->Enable_Override_Tip(d_settings->value(QStringLiteral("Override_Tip"), false).toBool());
	d_machine_dock->Enable_Machine_Errors(d_settings->value(QStringLiteral("Enable_Machine_Errors"), false).toBool());
	d_machine_dock->Set_Tip_Diameter(d_settings->value(QStringLiteral("Tip_Diameter"), 4.0).toDouble());
	d_machine_dock->Set_Noise(d_settings->value(QStringLiteral("Noise_Value"), 0.05).toDouble());
	d_machine_dock->Enable_Noise(d_settings->value(QStringLiteral("Generate_Noise"), false).toBool());
	d_machine_dock->Set_Temperature_X(d_settings->value(QStringLiteral("Machine_Temperature_X"), 20.0).toDouble());
	d_machine_dock->Set_Temperature_Y(d_settings->value(QStringLiteral("Machine_Temperature_Y"), 20.0).toDouble());
	d_machine_dock->Set_Temperature_Z(d_settings->value(QStringLiteral("Machine_Temperature_Z"), 20.0).toDouble());
	d_machine_dock->Set_Temperature_Part(d_settings->value(QStringLiteral("Machine_Temperature_Part"), 20.0).toDouble());
	
	d_machine_dock->Set_Touch_Buffer_Size(d_settings->value(QStringLiteral("Machine_Touch_Buffer_Size"), 5).toInt());
	d_communication_buffer_widget->Set_Buffer_Size(d_settings->value(QStringLiteral("Communication_Buffer_Size"), 16).toInt());
	
	d_machine_errors_dialog->Set_Scale_X(d_settings->value(QStringLiteral("Machine_Error_Scale_X"), 0.0).toDouble());
	d_machine_errors_dialog->Set_Scale_Y(d_settings->value(QStringLiteral("Machine_Error_Scale_Y"), 0.0).toDouble());
	d_machine_errors_dialog->Set_Scale_Z(d_settings->value(QStringLiteral("Machine_Error_Scale_Z"), 0.0).toDouble());
	d_machine_errors_dialog->Set_Squareness_XY(d_settings->value(QStringLiteral("Machine_Error_Squareness_XY"), 0.0).toDouble());
	d_machine_errors_dialog->Set_Squareness_YZ(d_settings->value(QStringLiteral("Machine_Error_Squareness_YZ"), 0.0).toDouble());
	d_machine_errors_dialog->Set_Squareness_ZX(d_settings->value(QStringLiteral("Machine_Error_Squareness_ZX"), 0.0).toDouble());
	d_machine_errors_dialog->Set_Correct_Ballbar(d_settings->value(QStringLiteral("Machine_Correct_Ballbar"), true).toBool());

	d_opengl_display_widget->Define_Machine_Volume(min_machine,max_machine);
	d_opengl_display_widget->Set_Touch_Buffer_Size(d_machine_dock->Touch_Buffer_Size());

	if(d_machine_dock->Override_Tip_Enabled())
	{
		d_coordinate_display_widget->Set_Tool_Tip_Diameter(d_machine_dock->Tip_Diameter());
		d_artifact_container_widget->Set_Tool_Tip_Diameter(d_machine_dock->Tip_Diameter());
	}
	
	d_artifact_container_widget->Load_FreeTouch_Points(d_data_path);
		
	logo_label->setPixmap(QPixmap(QStringLiteral(":/images/logo.png")));
	
	d_protocol_dialog->Reset();
	d_active_controller_type = d_protocol_dialog->Controller_Type();
	
	switch(d_active_controller_type)
	{
		case TProtocolDialog::CONTROLLER_VIRTUALCMM:
			this->Add_Log_Text(QStringLiteral("INF:  Protocol configured for Virtual CMM."));
			break;
			
		case TProtocolDialog::CONTROLLER_LEITZ:
			this->Add_Log_Text(QStringLiteral("INF:  Protocol configured for Leitz."));
			break;
			
		case TProtocolDialog::CONTROLLER_DC:
			this->Add_Log_Text(QStringLiteral("INF:  Protocol configured for DC."));
			break;
			
		case TProtocolDialog::CONTROLLER_IPP:
			this->Add_Log_Text(QStringLiteral("INF:  Protocol configured for I++ Server."));
			break;
	}
		
	if(TProtocolDialog::PROTOCOL_ETHERNET == d_protocol_dialog->Connection_Type())
	{
		d_machine_dock->Set_Serial_Connect_Button(false,QString("Connect: %1").arg(d_protocol_dialog->Comm_Port()));
		d_tcp_server->listen(QHostAddress(d_protocol_dialog->Ethernet_Address()),d_protocol_dialog->Ethernet_Port());
	}
	else
	{
		d_machine_dock->Set_Serial_Connect_Button(true,QString("Connect: %1").arg(d_protocol_dialog->Comm_Port()));
	}
	
	move_position = (min_machine + max_machine) / 2.0;
	d_coordinate_display_widget->Set_XYZ(move_position);
	
	// ballbar
	pos.x = d_settings->value(QStringLiteral("Ballbar_Attribute_Center_X"), 600.0).toDouble();
	pos.y = d_settings->value(QStringLiteral("Ballbar_Attribute_Center_Y"), 1100.0).toDouble();
	pos.z = d_settings->value(QStringLiteral("Ballbar_Attribute_Center_Z"), -550.0).toDouble();

	vec.i = d_settings->value(QStringLiteral("Ballbar_Attribute_Axis_I"), 0.0).toDouble();
	vec.j = d_settings->value(QStringLiteral("Ballbar_Attribute_Axis_J"), 0.0).toDouble();
	vec.k = d_settings->value(QStringLiteral("Ballbar_Attribute_Axis_K"), 1.0).toDouble();
	
	dval = d_settings->value(QStringLiteral("Ballbar_Attribute_Length"), 800.0).toDouble();

	d_artifact_container_widget->Set_Ballbar_Attribute_Position(pos);
	d_artifact_container_widget->Set_Ballbar_Attribute_Axis(vec);
	d_artifact_container_widget->Set_Ballbar_Attribute_Length(dval);
	
	// gaugeblock
	pos.x = d_settings->value(QStringLiteral("Gaugeblock_Attribute_Center_X"), 600.0).toDouble();
	pos.y = d_settings->value(QStringLiteral("Gaugeblock_Attribute_Center_Y"), 1100.0).toDouble();
	pos.z = d_settings->value(QStringLiteral("Gaugeblock_Attribute_Center_Z"), -550.0).toDouble();

	vec.i = d_settings->value(QStringLiteral("Gaugeblock_Attribute_Axis_I"), 1.0).toDouble();
	vec.j = d_settings->value(QStringLiteral("Gaugeblock_Attribute_Axis_J"), 0.0).toDouble();
	vec.k = d_settings->value(QStringLiteral("Gaugeblock_Attribute_Axis_K"), 0.0).toDouble();
	
	dval = d_settings->value(QStringLiteral("Gaugeblock_Attribute_Length"), 10.0).toDouble();

	d_artifact_container_widget->Set_Gaugeblock_Attribute_Position(pos);
	d_artifact_container_widget->Set_Gaugeblock_Attribute_Axis(vec);
	d_artifact_container_widget->Set_Gaugeblock_Attribute_Length(dval);
	
	// Ringgauge
	pos.x = d_settings->value(QStringLiteral("Ringgauge_Attribute_Center_X"), 600.0).toDouble();
	pos.y = d_settings->value(QStringLiteral("Ringgauge_Attribute_Center_Y"), 1100.0).toDouble();
	pos.z = d_settings->value(QStringLiteral("Ringgauge_Attribute_Center_Z"), -550.0).toDouble();

	vec.i = d_settings->value(QStringLiteral("Ringgauge_Attribute_Axis_I"), 0.0).toDouble();
	vec.j = d_settings->value(QStringLiteral("Ringgauge_Attribute_Axis_J"), 0.0).toDouble();
	vec.k = d_settings->value(QStringLiteral("Ringgauge_Attribute_Axis_K"), 1.0).toDouble();
	
	dval = d_settings->value(QStringLiteral("Ringgauge_Attribute_Diameter"), 25.0).toDouble();

	d_artifact_container_widget->Set_Ringgauge_Attribute_Position(pos);
	d_artifact_container_widget->Set_Ringgauge_Attribute_Axis(vec);
	d_artifact_container_widget->Set_Ringgauge_Attribute_Diameter(dval);
	d_artifact_container_widget->Set_Ringgauge_Attribute_ID(d_settings->value(QStringLiteral("Ringgauge_Attribute_ID"), true).toBool());

	// sphere
	pos.x = d_settings->value(QStringLiteral("Sphere_Attribute_Center_X"), 600.0).toDouble();
	pos.y = d_settings->value(QStringLiteral("Sphere_Attribute_Center_Y"), 1100.0).toDouble();
	pos.z = d_settings->value(QStringLiteral("Sphere_Attribute_Center_Z"), -550.0).toDouble();

	dval = d_settings->value(QStringLiteral("Sphere_Attribute_Diameter"), 25.0).toDouble();

	d_artifact_container_widget->Set_Sphere_Attribute_Position(pos);
	d_artifact_container_widget->Set_Sphere_Attribute_Diameter(dval);
	
	// Stepgauge
	pos.x = d_settings->value(QStringLiteral("Stepgauge_Attribute_Center_X"), 600.0).toDouble();
	pos.y = d_settings->value(QStringLiteral("Stepgauge_Attribute_Center_Y"), 1100.0).toDouble();
	pos.z = d_settings->value(QStringLiteral("Stepgauge_Attribute_Center_Z"), -550.0).toDouble();

	vec.i = d_settings->value(QStringLiteral("Stepgauge_Attribute_Axis_I"), 0.0).toDouble();
	vec.j = d_settings->value(QStringLiteral("Stepgauge_Attribute_Axis_J"), 1.0).toDouble();
	vec.k = d_settings->value(QStringLiteral("Stepgauge_Attribute_Axis_K"), 0.0).toDouble();
	
	dval = d_settings->value(QStringLiteral("Stepgauge_Attribute_Length"), 810.0).toDouble();

	d_artifact_container_widget->Set_Stepgauge_Attribute_Position(pos);
	d_artifact_container_widget->Set_Stepgauge_Attribute_Axis(vec);
	d_artifact_container_widget->Set_Stepgauge_Attribute_Length(dval);


	// default machine
	d_machine = new TMachineOffline;
	this->Initialize_Machine();
	
	d_machine_scale_factor_x = (d_machine_errors_dialog->Scale_X() + 1000.0) / 1000.0;
	d_machine_scale_factor_y = (d_machine_errors_dialog->Scale_Y() + 1000.0) / 1000.0;
	d_machine_scale_factor_z = (d_machine_errors_dialog->Scale_Z() + 1000.0) / 1000.0;
	
	d_machine_squareness_xy = d_machine_errors_dialog->Squareness_XY() / 1000.0;
	d_machine_squareness_yz = d_machine_errors_dialog->Squareness_YZ() / 1000.0;
	d_machine_squareness_zx = d_machine_errors_dialog->Squareness_ZX() / 1000.0;

	this->setWindowTitle(QStringLiteral("Virtual CMM"));
	touchpoint_dock->setWindowTitle(QStringLiteral("Touch Points"));
	
	copyright_label->setText(QStringLiteral("Copyright (C) 2026 Select Calibration Incorporated.  www.selectcalibration.ca"));
	
	d_file_quit_action->setText(QStringLiteral("Quit"));
	d_help_about_action->setText(QStringLiteral("About"));
	file_menu->setTitle(QStringLiteral("File"));
	view_menu->setTitle(QStringLiteral("View"));
	help_menu->setTitle(QStringLiteral("Help"));
		
	connect(d_help_about_action,&QAction::triggered,this,&TVirtualCMM::Menu_About);
	connect(d_file_quit_action,&QAction::triggered,this,&TVirtualCMM::close);

	connect(d_machine_dock,&TMachineDock::Attribute_Changed,this,&TVirtualCMM::Machine_Attribute_Changed);
	connect(d_machine_dock,&TMachineDock::Touch_Buffer_Changed,this,&TVirtualCMM::Machine_Touch_Buffer_Changed);
	connect(d_machine_dock,&TMachineDock::Show_Hide_Jogbox,this,&TVirtualCMM::Machine_Show_Hide_Jogbox);
	connect(d_machine_dock,&TMachineDock::Move,this,&TVirtualCMM::Machine_Move_To);
	
	connect(d_machine_dock,&TMachineDock::Enable_Machine_Errors_Changed,this,&TVirtualCMM::Machine_Enable_Errors_State_Changed);
	connect(d_machine_dock,&TMachineDock::Override_Tip_State_Changed,this,&TVirtualCMM::Machine_Override_Tip_State_Changed);
	connect(d_machine_dock,&TMachineDock::Override_Tip_Diameter_Changed,this,&TVirtualCMM::Machine_Override_Tip_Diameter_Changed);
	connect(d_machine_dock,&TMachineDock::Generate_Noise_State_Changed,this,&TVirtualCMM::Machine_Override_Noise_State_Changed);
	connect(d_machine_dock,&TMachineDock::Generate_Noise_Value_Changed,this,&TVirtualCMM::Machine_Override_Noise_Value_Changed);
	connect(d_machine_dock,&TMachineDock::Change_Protocol,this,&TVirtualCMM::Machine_Change_Protocol);
	connect(d_machine_dock,&TMachineDock::Edit_Machine_Errors,this,&TVirtualCMM::Machine_Edit_Errors);
	connect(d_machine_dock,&TMachineDock::Connect_Serial,this,&TVirtualCMM::New_Connection_Serial);
	
	connect(d_jogbox_dialog,&TJogboxDialog::MStart_Pressed,this,&TVirtualCMM::Jogbox_MStart_Pressed);
	connect(d_jogbox_dialog,&TJogboxDialog::DelPnt_Pressed,this,&TVirtualCMM::Jogbox_DelPnt_Pressed);
	connect(d_jogbox_dialog,&TJogboxDialog::Done_Pressed,this,&TVirtualCMM::Jogbox_Done_Pressed);
	connect(d_jogbox_dialog,&TJogboxDialog::EStop_Pressed,this,&TVirtualCMM::Jogbox_EStop_Pressed);
	connect(d_jogbox_dialog,&TJogboxDialog::Move_XM,this,&TVirtualCMM::Jogbox_Move_XM);
	connect(d_jogbox_dialog,&TJogboxDialog::Move_XP,this,&TVirtualCMM::Jogbox_Move_XP);
	connect(d_jogbox_dialog,&TJogboxDialog::Move_YM,this,&TVirtualCMM::Jogbox_Move_YM);
	connect(d_jogbox_dialog,&TJogboxDialog::Move_YP,this,&TVirtualCMM::Jogbox_Move_YP);
	connect(d_jogbox_dialog,&TJogboxDialog::Move_ZM,this,&TVirtualCMM::Jogbox_Move_ZM);
	connect(d_jogbox_dialog,&TJogboxDialog::Move_ZP,this,&TVirtualCMM::Jogbox_Move_ZP);

	connect(d_artifact_container_widget,&TArtifactContainerWidget::Touch,this,&TVirtualCMM::Machine_Add_Manual_Touch);
	connect(d_tcp_server,&QTcpServer::newConnection,this,&TVirtualCMM::New_Connection_Ethernet);
	connect(d_tcp_server,&QTcpServer::acceptError,this,&TVirtualCMM::Connection_Error);
	connect(d_serial_port,&QSerialPort::readyRead,this,&TVirtualCMM::Ready_Read);
}

TVirtualCMM::~TVirtualCMM(void)
{
	if(d_machine)
	{
		disconnect(d_machine,0,0,0);

		delete d_machine;
		d_machine = 0;
	}
}

void TVirtualCMM::Menu_About(void)
{
	d_msg_box->setText(QStringLiteral("Help - About"));
	d_msg_box->setInformativeText(QStringLiteral("Virtual CMM - Version 8.0"));
	d_msg_box->setDetailedText(QString());
	d_msg_box->setIcon(QMessageBox::Information);
	d_msg_box->setStandardButtons(QMessageBox::Ok);
	d_msg_box->setDefaultButton(QMessageBox::Ok);
	
	d_msg_box->exec();
}

void TVirtualCMM::Machine_Attribute_Changed(void)
{
	TVector3							min,max;
	
	min = d_machine_dock->Min_Machine();
	max = d_machine_dock->Max_Machine();
	
	d_opengl_display_widget->Define_Machine_Volume(min,max);
	d_opengl_display_widget->Update(false);
	
	if(d_machine)
	{
		d_machine->Set_Machine_Limits(min,max);
		
		d_machine->Set_Temperature_X(d_machine_dock->Temperature_X());
		d_machine->Set_Temperature_Y(d_machine_dock->Temperature_Y());
		d_machine->Set_Temperature_Z(d_machine_dock->Temperature_Z());
		d_machine->Set_Temperature_Part(d_machine_dock->Temperature_Part());
	}
}

void TVirtualCMM::Machine_Touch_Buffer_Changed(
	int									buffer_size)
{
	d_opengl_display_widget->Set_Touch_Buffer_Size(buffer_size);
	
	if(d_machine)
	{
		// force redraw
		d_opengl_display_widget->Move_To(d_machine->Position());
	}
}

void TVirtualCMM::Machine_Show_Hide_Jogbox(void)
{
	if(d_jogbox_dialog->isVisible())
	{
		d_jogbox_dialog->hide();
	}
	else
	{
		d_jogbox_dialog->show();
		d_jogbox_dialog->raise();
	}
}

void TVirtualCMM::Machine_Move_To(
	const double						&x,
	const double						&y,
	const double						&z)
{
	if(d_machine)
	{
		if(!d_machine->Move_To(TVector3(x,y,z)))
		{
			this->Add_Log_Text(d_machine->Last_Error());
		}
	}
}

void TVirtualCMM::Machine_Enable_Errors_State_Changed(
	const bool							state)
{
	if(state)
	{
		d_machine->Set_Scale_Factor_X(d_machine_scale_factor_x);
		d_machine->Set_Scale_Factor_Y(d_machine_scale_factor_y);
		d_machine->Set_Scale_Factor_Z(d_machine_scale_factor_z);
		d_machine->Set_Squareness_Error_XY(d_machine_squareness_xy);
		d_machine->Set_Squareness_Error_YZ(d_machine_squareness_yz);
		d_machine->Set_Squareness_Error_ZX(d_machine_squareness_zx);
	}
	else
	{
		d_machine->Set_Scale_Factor_X(1.0);
		d_machine->Set_Scale_Factor_Y(1.0);
		d_machine->Set_Scale_Factor_Z(1.0);
		d_machine->Set_Squareness_Error_XY(0.0);
		d_machine->Set_Squareness_Error_YZ(0.0);
		d_machine->Set_Squareness_Error_ZX(0.0);
	}
}

void TVirtualCMM::Machine_Override_Tip_State_Changed(
	const bool							state)
{	
	if(d_machine)
	{
		d_machine->Set_Override_Tip(state,d_machine_dock->Tip_Diameter());
		d_coordinate_display_widget->Set_Tool_Tip_Diameter(d_machine->Tip_Diameter());
		d_opengl_display_widget->Set_Tool_Tip_Diameter(d_machine->Tip_Diameter());
	}
}

void TVirtualCMM::Machine_Override_Tip_Diameter_Changed(
	const double						&diameter)
{	
	if(d_machine)
	{
		d_machine->Set_Override_Tip(d_machine_dock->Override_Tip_Enabled(),diameter);
		d_coordinate_display_widget->Set_Tool_Tip_Diameter(d_machine->Tip_Diameter());
		d_opengl_display_widget->Set_Tool_Tip_Diameter(d_machine->Tip_Diameter());
	}
}

void TVirtualCMM::Machine_Override_Noise_State_Changed(
	 const bool							state)
{
	if(d_machine)
	{
		d_machine->Set_Generate_Noise(state,d_machine_dock->Noise_Value());
	}
}

void TVirtualCMM::Machine_Override_Noise_Value_Changed(
	const double						&value)
{
	if(d_machine)
	{
		d_machine->Set_Generate_Noise(d_machine_dock->Generate_Noise_Enabled(),value);
	}
}

void TVirtualCMM::Machine_Position_Changed(
	const double						&x,
	const double						&y,
	const double						&z)
{
	TVector3							pos(x,y,z);
	QString								text;
	QByteArray							completion_text;
	
	if(d_machine)
	{
		d_opengl_display_widget->Move_To(pos);
		d_coordinate_display_widget->Set_XYZ(pos);
			
		d_opengl_display_widget->Update(false);
	}
}

void TVirtualCMM::Machine_Touch_Complete(
	const double						&x,
	const double						&y,
	const double						&z)
{
	TVector3							pos(x,y,z);
	
	d_opengl_display_widget->Add_Touch(pos);
}

void TVirtualCMM::Machine_Sensor_Changed(
	const double						&x,
	const double						&y,
	const double						&z,
	const double						&diameter)
{
	TVector3							offset(x,y,z);
	
	d_coordinate_display_widget->Set_Tool(offset,diameter);
	d_artifact_container_widget->Set_Tool(offset,diameter);
	d_opengl_display_widget->Set_Tool(offset,diameter);
}

void TVirtualCMM::Machine_Head_AB_Changed(
	const double						&a,
	const double						&b)
{
	d_opengl_display_widget->Set_Head_AB(a,b);
	d_coordinate_display_widget->Set_Head_AB(a,b);
	d_artifact_container_widget->Set_Head_AB(a,b);
}

void TVirtualCMM::Machine_Add_Manual_Touch(
	const double						&x,
	const double						&y,
	const double						&z,
	const double						&i,
	const double						&j,
	const double						&k,
	const bool							pre_compensate_ballbar)
{
	TVector3							pos(x,y,z);
	TVector3							vec(i,j,k);
	TVector3							pos_comp;

	if(d_machine)
	{
		if(pre_compensate_ballbar &&
		   d_machine_dock->Machine_Errors_Enabled() &&
		   d_machine_errors_dialog->Correct_Ballbar())
		{
			// kludge so that measurements such as ballbars end up with the expected length when machine errors are enabled.
			// artifacts do not really exist so each measurement grows or shrinks the length each time it is measured.
			// three stages are used (man, dcc, measurement) so the length is tweaked to remove two of the three
			
			// scale errors
			pos_comp.x = pos.x / d_machine_scale_factor_x;
			pos_comp.y = pos.y / d_machine_scale_factor_y;
			pos_comp.z = pos.z / d_machine_scale_factor_z;

			pos_comp.x /= d_machine_scale_factor_x;
			pos_comp.y /= d_machine_scale_factor_y;
			pos_comp.z /= d_machine_scale_factor_z;

			// squareness errors
			pos.x = pos_comp.x - (2.0 * pos_comp.y * d_machine_squareness_xy) - (2.0 * pos_comp.z * d_machine_squareness_zx);
			pos.y = pos_comp.y - (2.0 * pos_comp.z * d_machine_squareness_yz);
			pos.z = pos_comp.z;
		}
		
		if(!d_machine->Add_Manual_Touch_Point(pos,vec))
		{
			this->Add_Log_Text(d_machine->Last_Error());
		}
	}
}

void TVirtualCMM::Machine_Change_Protocol(void)
{
	d_protocol_dialog->Reset();
	
	if(QDialog::Accepted == d_protocol_dialog->exec())
	{
		d_active_controller_type = d_protocol_dialog->Controller_Type();
		
		switch(d_active_controller_type)
		{
			case TProtocolDialog::CONTROLLER_VIRTUALCMM:
				this->Add_Log_Text(QStringLiteral("INF:  Protocol configured for Virtual CMM."));
				break;
				
			case TProtocolDialog::CONTROLLER_LEITZ:
				this->Add_Log_Text(QStringLiteral("INF:  Protocol configured for Leitz."));
				break;
				
			case TProtocolDialog::CONTROLLER_DC:
				this->Add_Log_Text(QStringLiteral("INF:  Protocol configured for DC."));
				break;
				
			case TProtocolDialog::CONTROLLER_IPP:
				this->Add_Log_Text(QStringLiteral("INF:  Protocol configured for I++ Server."));
				break;
		}
		
		if(TProtocolDialog::PROTOCOL_ETHERNET == d_protocol_dialog->Connection_Type())
		{
			d_machine_dock->Set_Serial_Connect_Button(false,QString("Connect: %1").arg(d_protocol_dialog->Comm_Port()));
			
			d_tcp_server->close();
			
			while(d_tcp_server->isListening())
			{
				QCoreApplication::processEvents();
				QThread::msleep(100);
			}
			
			d_tcp_server->listen(QHostAddress(d_protocol_dialog->Ethernet_Address()),d_protocol_dialog->Ethernet_Port());
		}
		else
		{
			d_machine_dock->Set_Serial_Connect_Button(true,QString("Connect: %1").arg(d_protocol_dialog->Comm_Port()));
			d_tcp_server->close();
		}
	}
}

void TVirtualCMM::Machine_Edit_Errors(void)
{
	if(QDialog::Accepted == d_machine_errors_dialog->exec())
	{
		d_machine_scale_factor_x = (d_machine_errors_dialog->Scale_X() + 1000.0) / 1000.0;
		d_machine_scale_factor_y = (d_machine_errors_dialog->Scale_Y() + 1000.0) / 1000.0;
		d_machine_scale_factor_z = (d_machine_errors_dialog->Scale_Z() + 1000.0) / 1000.0;
		
		d_machine_squareness_xy = d_machine_errors_dialog->Squareness_XY() / 1000.0;
		d_machine_squareness_yz = d_machine_errors_dialog->Squareness_YZ() / 1000.0;
		d_machine_squareness_zx = d_machine_errors_dialog->Squareness_ZX() / 1000.0;
		
		if(d_machine_dock->Machine_Errors_Enabled())
		{
			d_machine->Set_Scale_Factor_X(d_machine_scale_factor_x);
			d_machine->Set_Scale_Factor_Y(d_machine_scale_factor_y);
			d_machine->Set_Scale_Factor_Z(d_machine_scale_factor_z);
			d_machine->Set_Squareness_Error_XY(d_machine_squareness_xy);
			d_machine->Set_Squareness_Error_YZ(d_machine_squareness_yz);
			d_machine->Set_Squareness_Error_ZX(d_machine_squareness_zx);
		}
		else
		{
			d_machine->Set_Scale_Factor_X(1.0);
			d_machine->Set_Scale_Factor_Y(1.0);
			d_machine->Set_Scale_Factor_Z(1.0);
			d_machine->Set_Squareness_Error_XY(0.0);
			d_machine->Set_Squareness_Error_YZ(0.0);
			d_machine->Set_Squareness_Error_ZX(0.0);
		}
	}
}

void TVirtualCMM::Jogbox_MStart_Pressed(void)
{
	this->Add_Log_Text(QStringLiteral("INF:  Machine Online"));
	
	if(d_machine)
	{
		d_machine->Set_EStop(d_jogbox_dialog->EStop());
	}
}

void TVirtualCMM::Jogbox_DelPnt_Pressed(void)
{
	this->Add_Log_Text(QStringLiteral("INF:  Keypress Erase Hit"));
	
	if(d_machine)
	{
		this->Send(d_machine->KeyPress_DelPnt());
	}
}

void TVirtualCMM::Jogbox_Done_Pressed(void)
{
	this->Add_Log_Text(QStringLiteral("INF:  Keypress Done"));
	
	if(d_machine)
	{		
		this->Send(d_machine->KeyPress_Done());
	}
}

void TVirtualCMM::Jogbox_EStop_Pressed(void)
{
	this->Add_Log_Text(QStringLiteral("WAR:  Machine in E-Stop"));

	if(d_machine)
	{
		d_machine->Set_EStop(d_jogbox_dialog->EStop());
	}
}

void TVirtualCMM::Jogbox_Move_XM(void)
{
	if(d_machine)
	{
		if(!d_machine->Jog_Move(TVector3(-1.0,0.0,0.0)))
		{
			this->Add_Log_Text(d_machine->Last_Error());
		}
	}
}

void TVirtualCMM::Jogbox_Move_XP(void)
{
	if(d_machine)
	{
		if(!d_machine->Jog_Move(TVector3(1.0,0.0,0.0)))
		{
			this->Add_Log_Text(d_machine->Last_Error());
		}
	}
}

void TVirtualCMM::Jogbox_Move_YM(void)
{
	if(d_machine)
	{
		if(!d_machine->Jog_Move(TVector3(0.0,-1.0,0.0)))
		{
			this->Add_Log_Text(d_machine->Last_Error());
		}
	}
}

void TVirtualCMM::Jogbox_Move_YP(void)
{
	if(d_machine)
	{
		if(!d_machine->Jog_Move(TVector3(0.0,1.0,0.0)))
		{
			this->Add_Log_Text(d_machine->Last_Error());
		}
	}
}

void TVirtualCMM::Jogbox_Move_ZM(void)
{
	if(d_machine)
	{
		if(!d_machine->Jog_Move(TVector3(0.0,0.0,-1.0)))
		{
			this->Add_Log_Text(d_machine->Last_Error());
		}
	}
}

void TVirtualCMM::Jogbox_Move_ZP(void)
{
	if(d_machine)
	{
		if(!d_machine->Jog_Move(TVector3(0.0,0.0,1.0)))
		{
			this->Add_Log_Text(d_machine->Last_Error());
		}
	}
}

void TVirtualCMM::New_Connection_Serial(void)
{	
	if(d_tcp_socket)
	{
		this->Add_Log_Text(QStringLiteral("ERR:  Ethernet connection is active."));
		return;
	}
	
	if(d_machine)
	{
		disconnect(d_machine,0,0,0);
		
		delete d_machine;
		d_machine = 0;
	}
	
	if(d_serial_port->isOpen())
	{
		d_serial_port->close();
		
		this->Add_Log_Text(QString("INF:  Comm port %1 closed.").arg(d_protocol_dialog->Comm_Port()));
		
		d_machine_dock->Enable_Protocol(true);
		d_machine_dock->Set_Serial_Connect_Button(true,QString("Connect: %1").arg(d_protocol_dialog->Comm_Port()));
		
	}
	else
	{
		d_serial_port->setPortName(d_protocol_dialog->Comm_Port());
		d_serial_port->setBaudRate(static_cast<int>(d_protocol_dialog->Baud_Rate()));
		
		d_serial_port->setParity(QSerialPort::NoParity);
		d_serial_port->setFlowControl(QSerialPort::NoFlowControl);
		
		if(!d_serial_port->open(QIODevice::ReadOnly | QIODevice::WriteOnly))
		{
			this->Add_Log_Text(QString("ERR:  Cannot open comm port %1.  Reason: %2").arg(d_protocol_dialog->Comm_Port()).arg(d_serial_port->errorString()));
		}
		else
		{
			this->Add_Log_Text(QString("INF:  Comm port %1 open.").arg(d_protocol_dialog->Comm_Port()));
			
			d_machine_dock->Enable_Protocol(false);
			d_machine_dock->Set_Serial_Connect_Button(true,QString("Disconnect: %1").arg(d_protocol_dialog->Comm_Port()));
			
			switch(d_active_controller_type)
			{
				case TProtocolDialog::CONTROLLER_VIRTUALCMM:
				case TProtocolDialog::CONTROLLER_LEITZ:
					d_machine = new TMachineLeitz;
					break;
					
				case TProtocolDialog::CONTROLLER_DC:
					d_machine = new TMachineDC;
					break;
					
				case TProtocolDialog::CONTROLLER_IPP:
					d_machine = new TMachineIpp;
					break;
			}
			
			this->Initialize_Machine();
		}
	}
	
	if(!d_machine)
	{
		d_machine = new TMachineOffline;
		this->Initialize_Machine();
	}
}

void TVirtualCMM::New_Connection_Ethernet(void)
{
	if(d_machine)
	{
		disconnect(d_machine,0,0,0);
		
		delete d_machine;
		d_machine = 0;
	}

	if(!d_tcp_socket)
	{		
		d_tcp_socket = d_tcp_server->nextPendingConnection();
		
		if(d_serial_port->isOpen())
		{
			this->Add_Log_Text(QStringLiteral("ERR:  Ethernet connection not allowed while serial is active."));

			// serial is active
			
			if(d_tcp_socket)
			{
				d_tcp_socket->close();
				d_tcp_socket = 0;
			}
			
			d_machine = new TMachineOffline;
			this->Initialize_Machine();
			
			return;
		}
		
		if(d_tcp_socket)
		{
			connect(d_tcp_socket,&QTcpSocket::disconnected,this,&TVirtualCMM::Closed_Connection);
			connect(d_tcp_socket,&QTcpSocket::readyRead,this,&TVirtualCMM::Ready_Read);
			
			this->Clear_Log();
			this->Add_Log_Text(QStringLiteral("INF:  Remote connection accepted."));
			
			d_machine_dock->Set_Serial_Connect_Button(false);
			d_machine_dock->Enable_Protocol(false);
			
			switch(d_active_controller_type)
			{
				case TProtocolDialog::CONTROLLER_VIRTUALCMM:
				case TProtocolDialog::CONTROLLER_LEITZ:
					d_machine = new TMachineLeitz;
					break;
					
				case TProtocolDialog::CONTROLLER_DC:
					d_machine = new TMachineDC;
					break;
					
				case TProtocolDialog::CONTROLLER_IPP:
					d_machine = new TMachineIpp;
					break;
			}
			
			this->Initialize_Machine();
		}
	}
	
	if(!d_machine)
	{
		d_machine = new TMachineOffline;
		this->Initialize_Machine();
	}
}

void TVirtualCMM::Closed_Connection(void)
{
	QTcpSocket							*socket;
	std::vector<QTcpSocket*>::iterator	iter;
	
	if(d_machine)
	{
		disconnect(d_machine,0,0,0);
		
		delete d_machine;
		d_machine = 0;
	}

	socket = static_cast<QTcpSocket*>(QObject::sender());
	
	if(socket && d_tcp_socket == socket)
	{
		this->Add_Log_Text(QStringLiteral("INF:  Remote connection closed."));

		d_tcp_socket->close();
		d_tcp_socket = 0;
		
		
		d_machine_dock->Enable_Protocol(true);
	}
	
	d_machine = new TMachineOffline;
	this->Initialize_Machine();
}

void TVirtualCMM::Connection_Error(
	QAbstractSocket::SocketError		socket_error)
{
	switch(socket_error)
	{
		case QAbstractSocket::RemoteHostClosedError:
			this->Add_Log_Text(QStringLiteral("ERR:  Connection closed remotely."));
			break;
			
		case QAbstractSocket::HostNotFoundError:
			this->Add_Log_Text(QStringLiteral("ERR:  Host not found."));
			break;
			
		case QAbstractSocket::SocketTimeoutError:
			this->Add_Log_Text(QStringLiteral("ERR:  Connection timeout."));
			break;
			
		case QAbstractSocket::ConnectionRefusedError:
			this->Add_Log_Text(QStringLiteral("ERR:  Connection refused."));
			break;
			
		default:
			this->Add_Log_Text(QString("ERR:  Socket error number: %1.").arg(static_cast<int>(socket_error)));
			break;
	}
}

void TVirtualCMM::Ready_Read(void)
{
	QByteArray							new_data;
	
	if(d_serial_port->isOpen())
	{
		new_data = d_serial_port->readAll();
	}
	else if(d_tcp_socket)
	{
		new_data = d_tcp_socket->readAll();
		
	}
	else
	{
		return;
	}
	
	if(d_machine)
	{
		d_machine->Process_Rx(new_data);
	}
}

void TVirtualCMM::Send(
	const QByteArray					&bytes)
{
	QString								write_text(QString(bytes).trimmed());
	
	if(d_machine)
	{
		if(d_serial_port->isOpen())
		{
			d_serial_port->write(bytes);
		}
		else if(d_tcp_socket)
		{
			d_tcp_socket->write(bytes);
		}
	}
	
	d_communication_buffer_widget->Add_Entry(TBufferDisplayWidget::BUFFER_SEND,write_text);
}

void TVirtualCMM::Log_Rx(
	const QString						&text)
{
	d_communication_buffer_widget->Add_Entry(TBufferDisplayWidget::BUFFER_RECEIVE,text);
}

void TVirtualCMM::Log_Text(
	const QString						&text)
{
	this->Add_Log_Text(text);
}

void TVirtualCMM::closeEvent(
	QCloseEvent							*event)
{
	QList<int>							splitter_sizes;
	TVector3							min_machine;
	TVector3							max_machine;
	TVector3							move_position;
	TVector3							pos;
	TVector3							axis;
	double								dval;
	
	if(d_tcp_socket != 0 || d_serial_port->isOpen() == true)
	{	
		d_msg_box->setText(QStringLiteral("Connection Active"));
		d_msg_box->setInformativeText(QStringLiteral("Communication connection still active.  Close anyway?"));
		d_msg_box->setDetailedText(QString());
		d_msg_box->setIcon(QMessageBox::Information);
		d_msg_box->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
		d_msg_box->setDefaultButton(QMessageBox::No);
		
		if(QMessageBox::No == d_msg_box->exec())
		{
			event->ignore();
			return;
		}
	}
	
	if(d_tcp_socket)
	{
		d_tcp_socket->close();
		d_tcp_socket = 0;
	}
	
	d_tcp_server->close();
	
	if(d_serial_port->isOpen())
	{
		d_serial_port->close();
	}
		
	splitter_sizes = d_splitter_widget->sizes();

	if(splitter_sizes.size() == 2)
	{
		d_settings->setValue(QStringLiteral("Graphic_Window_Height"), splitter_sizes[0]);
		d_settings->setValue(QStringLiteral("Information_Window_Height"), splitter_sizes[1]);
	}
	
	d_settings->setValue(QStringLiteral("Machine_Touch_Buffer_Size"), d_machine_dock->Touch_Buffer_Size());
	d_settings->setValue(QStringLiteral("Communication_Buffer_Size"), d_communication_buffer_widget->Buffer_Size());
	
	d_artifact_container_widget->Save_FreeTouch_Points(d_data_path);
	
	// ballbar
	pos = d_artifact_container_widget->Ballbar_Attribute_Position();
	axis = d_artifact_container_widget->Ballbar_Attribute_Axis();
	dval = d_artifact_container_widget->Ballbar_Attribute_Length();
	
	d_settings->setValue(QStringLiteral("Ballbar_Attribute_Center_X"), pos.x);
	d_settings->setValue(QStringLiteral("Ballbar_Attribute_Center_Y"), pos.y);
	d_settings->setValue(QStringLiteral("Ballbar_Attribute_Center_Z"), pos.z);
	
	d_settings->setValue(QStringLiteral("Ballbar_Attribute_Axis_I"), axis.x);
	d_settings->setValue(QStringLiteral("Ballbar_Attribute_Axis_J"), axis.y);
	d_settings->setValue(QStringLiteral("Ballbar_Attribute_Axis_K"), axis.z);
	
	d_settings->setValue(QStringLiteral("Ballbar_Attribute_Length"), dval);
	
	// gaugeblock
	pos = d_artifact_container_widget->Gaugeblock_Attribute_Position();
	axis = d_artifact_container_widget->Gaugeblock_Attribute_Axis();
	dval = d_artifact_container_widget->Gaugeblock_Attribute_Length();
	
	d_settings->setValue(QStringLiteral("Gaugeblock_Attribute_Center_X"), pos.x);
	d_settings->setValue(QStringLiteral("Gaugeblock_Attribute_Center_Y"), pos.y);
	d_settings->setValue(QStringLiteral("Gaugeblock_Attribute_Center_Z"), pos.z);
	
	d_settings->setValue(QStringLiteral("Gaugeblock_Attribute_Axis_I"), axis.x);
	d_settings->setValue(QStringLiteral("Gaugeblock_Attribute_Axis_J"), axis.y);
	d_settings->setValue(QStringLiteral("Gaugeblock_Attribute_Axis_K"), axis.z);
	
	d_settings->setValue(QStringLiteral("Gaugeblock_Attribute_Length"), dval);
	
	// Ringgauge
	pos = d_artifact_container_widget->Ringgauge_Attribute_Position();
	axis = d_artifact_container_widget->Ringgauge_Attribute_Axis();
	dval = d_artifact_container_widget->Ringgauge_Attribute_Diameter();
	
	d_settings->setValue(QStringLiteral("Ringgauge_Attribute_Center_X"), pos.x);
	d_settings->setValue(QStringLiteral("Ringgauge_Attribute_Center_Y"), pos.y);
	d_settings->setValue(QStringLiteral("Ringgauge_Attribute_Center_Z"), pos.z);
	
	d_settings->setValue(QStringLiteral("Ringgauge_Attribute_Axis_I"), axis.x);
	d_settings->setValue(QStringLiteral("Ringgauge_Attribute_Axis_J"), axis.y);
	d_settings->setValue(QStringLiteral("Ringgauge_Attribute_Axis_K"), axis.z);
	
	d_settings->setValue(QStringLiteral("Ringgauge_Attribute_Diameter"), dval);
	d_settings->setValue(QStringLiteral("Ringgauge_Attribute_ID"),d_artifact_container_widget->Ringgauge_Attribute_ID());
	
	// sphere
	pos = d_artifact_container_widget->Sphere_Attribute_Position();
	dval = d_artifact_container_widget->Ringgauge_Attribute_Diameter();
	
	d_settings->setValue(QStringLiteral("Sphere_Attribute_Center_X"), pos.x);
	d_settings->setValue(QStringLiteral("Sphere_Attribute_Center_Y"), pos.y);
	d_settings->setValue(QStringLiteral("Sphere_Attribute_Center_Z"), pos.z);
	
	d_settings->setValue(QStringLiteral("Sphere_Attribute_Diameter"), dval);
	
	// Stepgauge
	pos = d_artifact_container_widget->Stepgauge_Attribute_Position();
	axis = d_artifact_container_widget->Stepgauge_Attribute_Axis();
	dval = d_artifact_container_widget->Stepgauge_Attribute_Length();
	
	d_settings->setValue(QStringLiteral("Stepgauge_Attribute_Center_X"), pos.x);
	d_settings->setValue(QStringLiteral("Stepgauge_Attribute_Center_Y"), pos.y);
	d_settings->setValue(QStringLiteral("Stepgauge_Attribute_Center_Z"), pos.z);
	
	d_settings->setValue(QStringLiteral("Stepgauge_Attribute_Axis_I"), axis.x);
	d_settings->setValue(QStringLiteral("Stepgauge_Attribute_Axis_J"), axis.y);
	d_settings->setValue(QStringLiteral("Stepgauge_Attribute_Axis_K"), axis.z);
	
	d_settings->setValue(QStringLiteral("Stepgauge_Attribute_Length"), dval);

	// CMM Settings
	d_settings->setValue(QStringLiteral("Controller_Type"), static_cast<int>(d_protocol_dialog->Controller_Type()));
	d_settings->setValue(QStringLiteral("Connection_Type"), static_cast<int>(d_protocol_dialog->Connection_Type()));
	d_settings->setValue(QStringLiteral("Connection_Serial_Name"), d_protocol_dialog->Comm_Port());
	d_settings->setValue(QStringLiteral("Connection_Baud_Rate"), static_cast<int>(d_protocol_dialog->Baud_Rate()));
	d_settings->setValue(QStringLiteral("Connection_Ethernet_Address"), d_protocol_dialog->Ethernet_Address());
	d_settings->setValue(QStringLiteral("Connection_Ethernet_Port"), d_protocol_dialog->Ethernet_Port());

	// Machine settings
	min_machine = d_machine_dock->Min_Machine();
	max_machine = d_machine_dock->Max_Machine();
	move_position = d_machine_dock->Move_Position();

	d_settings->setValue(QStringLiteral("Machine_Min_X"), min_machine.x);
	d_settings->setValue(QStringLiteral("Machine_Min_Y"), min_machine.y);
	d_settings->setValue(QStringLiteral("Machine_Min_Z"), min_machine.z);

	d_settings->setValue(QStringLiteral("Machine_Max_X"), max_machine.x);
	d_settings->setValue(QStringLiteral("Machine_Max_Y"), max_machine.y);
	d_settings->setValue(QStringLiteral("Machine_Max_Z"), max_machine.z);
	
	d_settings->setValue(QStringLiteral("Move_Position_X"), move_position.x);
	d_settings->setValue(QStringLiteral("Move_Position_Y"), move_position.y);
	d_settings->setValue(QStringLiteral("Move_Position_Z"), move_position.z);

	d_settings->setValue(QStringLiteral("Tip_Diameter"), d_machine_dock->Tip_Diameter());
	d_settings->setValue(QStringLiteral("Override_Tip"), d_machine_dock->Override_Tip_Enabled());

	d_settings->setValue(QStringLiteral("Noise_Value"), d_machine_dock->Noise_Value());
	d_settings->setValue(QStringLiteral("Generate_Noise"), d_machine_dock->Generate_Noise_Enabled());
	
	d_settings->setValue(QStringLiteral("Machine_Temperature_X"), d_machine_dock->Temperature_X());
	d_settings->setValue(QStringLiteral("Machine_Temperature_Y"), d_machine_dock->Temperature_Y());
	d_settings->setValue(QStringLiteral("Machine_Temperature_Z"), d_machine_dock->Temperature_Z());
	d_settings->setValue(QStringLiteral("Machine_Temperature_Part"), d_machine_dock->Temperature_Part());

	d_settings->setValue(QStringLiteral("Enable_Machine_Errors"), d_machine_dock->Machine_Errors_Enabled());
	d_settings->setValue(QStringLiteral("Machine_Error_Scale_X"), d_machine_errors_dialog->Scale_X());
	d_settings->setValue(QStringLiteral("Machine_Error_Scale_Y"), d_machine_errors_dialog->Scale_Y());
	d_settings->setValue(QStringLiteral("Machine_Error_Scale_Z"), d_machine_errors_dialog->Scale_Z());
	d_settings->setValue(QStringLiteral("Machine_Error_Squareness_XY"), d_machine_errors_dialog->Squareness_XY());
	d_settings->setValue(QStringLiteral("Machine_Error_Squareness_YZ"), d_machine_errors_dialog->Squareness_YZ());
	d_settings->setValue(QStringLiteral("Machine_Error_Squareness_ZX"), d_machine_errors_dialog->Squareness_ZX());
	d_settings->setValue(QStringLiteral("Machine_Correct_Ballbar"), d_machine_errors_dialog->Correct_Ballbar());

	// main window
	d_settings->setValue(QStringLiteral("Mainwindow_Position"), this->pos());
	d_settings->setValue(QStringLiteral("Mainwindow_Size"), this->size());

	event->accept();
}

void TVirtualCMM::Initialize_Machine(void)
{
	if(d_machine)
	{
		d_machine->Set_Machine_Limits(d_machine_dock->Min_Machine(),d_machine_dock->Max_Machine());
		d_machine->Set_Tool_Offset(d_coordinate_display_widget->Tool_Offset());
		d_machine->Set_Head_AB(d_coordinate_display_widget->Head_A(),d_coordinate_display_widget->Head_B());
		d_machine->Set_Override_Tip(d_machine_dock->Override_Tip_Enabled(),d_machine_dock->Tip_Diameter());
		d_machine->Set_Generate_Noise(d_machine_dock->Generate_Noise_Enabled(),d_machine_dock->Noise_Value());
		d_machine->Set_Current_Position(d_coordinate_display_widget->Position());
		d_machine->Set_EStop(d_jogbox_dialog->EStop());
		
		if(d_machine_dock->Machine_Errors_Enabled())
		{
			d_machine->Set_Scale_Factor_X(d_machine_scale_factor_x);
			d_machine->Set_Scale_Factor_Y(d_machine_scale_factor_y);
			d_machine->Set_Scale_Factor_Z(d_machine_scale_factor_z);
			d_machine->Set_Squareness_Error_XY(d_machine_squareness_xy);
			d_machine->Set_Squareness_Error_YZ(d_machine_squareness_yz);
			d_machine->Set_Squareness_Error_ZX(d_machine_squareness_zx);
		}
		else
		{
			d_machine->Set_Scale_Factor_X(1.0);
			d_machine->Set_Scale_Factor_Y(1.0);
			d_machine->Set_Scale_Factor_Z(1.0);
			d_machine->Set_Squareness_Error_XY(0.0);
			d_machine->Set_Squareness_Error_YZ(0.0);
			d_machine->Set_Squareness_Error_ZX(0.0);
		}
		
		d_machine->Set_Temperature_X(d_machine_dock->Temperature_X());
		d_machine->Set_Temperature_Y(d_machine_dock->Temperature_Y());
		d_machine->Set_Temperature_Z(d_machine_dock->Temperature_Z());
		d_machine->Set_Temperature_Part(d_machine_dock->Temperature_Part());

		d_artifact_container_widget->Set_Tool(d_coordinate_display_widget->Tool_Offset(),d_coordinate_display_widget->Tip_Diameter());
				
		d_opengl_display_widget->Set_Tool(d_coordinate_display_widget->Tool_Offset(),d_coordinate_display_widget->Tip_Diameter());
		d_opengl_display_widget->Set_Head_AB(d_coordinate_display_widget->Head_A(),d_coordinate_display_widget->Head_B());
		d_opengl_display_widget->Move_To(d_machine->Position());
		
		this->Add_Log_Text(QStringLiteral("INF:  Machine type:  ") + d_machine->Machine_Description());
		
		connect(d_machine,&TMachine::Position_Changed,this,&TVirtualCMM::Machine_Position_Changed);
		connect(d_machine,&TMachine::Touch_Complete,this,&TVirtualCMM::Machine_Touch_Complete);
		connect(d_machine,&TMachine::Write_Data,this,&TVirtualCMM::Send);
		connect(d_machine,&TMachine::Log_Rx,this,&TVirtualCMM::Log_Rx);
		connect(d_machine,&TMachine::Log_Text,this,&TVirtualCMM::Log_Text);
		connect(d_machine,&TMachine::Tool_Offset_Changed,this,&TVirtualCMM::Machine_Sensor_Changed);
		connect(d_machine,&TMachine::Head_AB_Changed,this,&TVirtualCMM::Machine_Head_AB_Changed);
	}
}

void TVirtualCMM::Clear_Log(void)
{
	d_message_log->clear();
}

void TVirtualCMM::Add_Log_Text(
	const QString						&text)
{	
	d_message_log->moveCursor(QTextCursor::End);
	d_message_log->appendPlainText(text);
}

