/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QFont>
#include <QFontMetrics>
#include <QFrame>
#include <QGridLayout>
#include <QLabel>
#include <QLineEdit>
#include <QSizePolicy>
#include <QSpacerItem>
#include <QToolButton>
#include <QPixmap>
#include <QPlainTextEdit>
#include <QTextCursor>
#include <QIcon>
#include <QSettings>
#include <QHostAddress>
#include <QTcpSocket>
#include <QCloseEvent>
#include <QDir>
#include <QSize>
#include <QThread>
#include <QTimer>
#include <QFile>
#include <QFileDialog>

#include "dccontrollerstatus.h"

static const int						OUTPUT_FIELD_TITLE_WIDTH(-20);

TDCControllerStatus::TDCControllerStatus(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QDialog(const_cast<QWidget*>(parent),flags)
{
	QFont								title_font;
	QFont								text_font;
	QFont								copyright_font;
	QFrame								*separator_hline1;
	QFrame								*separator_hline2;
	QGridLayout							*dialog_layout;
	QGridLayout							*status_layout;
	QLabel								*controller_connection_label;
	QLabel								*controller_ip_label;
	QLabel								*controller_port_label;
	QLabel								*copyright_label;
	QLabel								*error_history_label;
	QLabel								*communication_log_label;
	QLabel								*logo_label;
	QLabel								*machine_status_label;
	QLabel								*relays_label;
	QLabel								*estop_label;
	QLabel								*autzer_label;
	QLabel								*prbtype_label;
	QLabel								*prbstatus_label;
	QLabel								*machine_type_label;
	QLabel								*machine_serial_label;
	QLabel								*fw_version_label;
	QLabel								*tempcomptype_label;
	QLabel								*volcomp_type_label;
	QLabel								*volcomp_label;
	QLabel								*geo_label;
	QLabel								*mapstat_label;
	QLabel								*dyncomp_label;
	QLabel								*x_cte_label;
	QLabel								*y_cte_label;
	QLabel								*z_cte_label;
	QLabel								*x_temperature_label;
	QLabel								*y_temperature_label;
	QLabel								*z_temperature_label;
	QLabel								*w_temperature_label;
	QLabel								*part_temperature_label;
	QSizePolicy							size_policy_minimum_fixed(QSizePolicy::Minimum, QSizePolicy::Fixed);
	QSizePolicy							size_policy_minimum_minimum(QSizePolicy::Minimum,QSizePolicy::Minimum);
	QSizePolicy							size_policy_maximum_preferred(QSizePolicy::Maximum, QSizePolicy::Preferred);
	QSizePolicy							size_policy_minexp_preferred(QSizePolicy::MinimumExpanding, QSizePolicy::Preferred);
	QSpacerItem							*connection_hspacer;
	QDir								data_path;

	this->resize(801,801);

	title_font.setWeight(75);
	title_font.setItalic(true);
	title_font.setBold(true);
	title_font.setUnderline(true);

	text_font.setFamily(QStringLiteral("Courier New"));
	text_font.setFixedPitch(true);
	
	size_policy_minimum_fixed.setHorizontalStretch(0);
	size_policy_minimum_fixed.setVerticalStretch(0);

	size_policy_minimum_minimum.setHorizontalStretch(0);
	size_policy_minimum_minimum.setVerticalStretch(0);

	size_policy_maximum_preferred.setHorizontalStretch(0);
	size_policy_maximum_preferred.setVerticalStretch(0);

	size_policy_minexp_preferred.setHorizontalStretch(0);
	size_policy_minexp_preferred.setVerticalStretch(0);

	copyright_font.setPointSize(8);
	copyright_font.setItalic(true);
	
	d_application_path = qApp->applicationDirPath();
	
#ifdef Q_OS_MAC
	int index = d_application_path.lastIndexOf(QStringLiteral(".app/"));
	d_application_path.truncate(index);
	
	index = d_application_path.lastIndexOf('/');
	d_application_path.truncate(index);
#endif
	
#ifdef Q_OS_WIN
	d_data_path = d_application_path;
#else
	d_data_path = QDir::homePath();
#endif
	
	data_path.setPath(d_data_path);
	
	if(!data_path.exists(QStringLiteral(".dccontrollerstatus")))
	{
		data_path.mkdir(QStringLiteral(".dccontrollerstatus"));
	}
	
	data_path.cd(QStringLiteral(".dccontrollerstatus"));
	d_data_path = data_path.absolutePath();
	
	d_settings = new QSettings(d_data_path + QStringLiteral("/dccontrollerstatus_settings.ini"),QSettings::IniFormat,this);

	dialog_layout = new QGridLayout(this);

	controller_connection_label = new QLabel(this);
	controller_connection_label->setFont(title_font);
	dialog_layout->addWidget(controller_connection_label,0,0,1,5);

	controller_ip_label = new QLabel(this);
	dialog_layout->addWidget(controller_ip_label,1,0,1,1);

	d_controller_ip_edit = new QLineEdit(this);
	d_controller_ip_edit->setAlignment(Qt::AlignCenter);
	d_controller_ip_edit->setSizePolicy(size_policy_minimum_fixed);
	dialog_layout->addWidget(d_controller_ip_edit,1,1,1,1);

	connection_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	dialog_layout->addItem(connection_hspacer,1,2,2,1);

	controller_port_label = new QLabel(this);
	dialog_layout->addWidget(controller_port_label,2,0,1,1);

	d_controller_port_edit = new QLineEdit(this);
	d_controller_port_edit->setAlignment(Qt::AlignCenter);
	d_controller_port_edit->setSizePolicy(size_policy_minimum_fixed);
	dialog_layout->addWidget(d_controller_port_edit,2,1,1,1);
	
	d_update_button = new QToolButton(this);
	d_update_button->setMinimumSize(48,48);
	d_update_button->setIconSize(QSize(48,48));
	d_update_button->setSizePolicy(size_policy_minimum_minimum);
	dialog_layout->addWidget(d_update_button,1,3,2,1);
	
	d_save_button = new QToolButton(this);
	d_save_button->setMinimumSize(48,48);
	d_save_button->setIconSize(QSize(48,48));
	d_save_button->setSizePolicy(size_policy_minimum_minimum);
	dialog_layout->addWidget(d_save_button,1,4,2,1);

	separator_hline1 = new QFrame(this);
	separator_hline1->setFrameShape(QFrame::HLine);
	separator_hline1->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(separator_hline1,3,0,1,5);

	machine_status_label = new QLabel(this);
	machine_status_label->setFont(title_font);
	dialog_layout->addWidget(machine_status_label,4,0,1,5);
	
	status_layout = new QGridLayout();
	status_layout->setSpacing(4);
	
	relays_label = new QLabel(this);
	relays_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	relays_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(relays_label,0,0,1,1);
	
	d_relays_label = new QLabel(this);
	d_relays_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_relays_label,0,1,1,1);

	estop_label = new QLabel(this);
	estop_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	estop_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(estop_label,1,0,1,1);
	
	d_estop_label = new QLabel(this);
	d_estop_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_estop_label,1,1,1,1);

	autzer_label = new QLabel(this);
	autzer_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	autzer_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(autzer_label,2,0,1,1);
	
	d_autzer_label = new QLabel(this);
	d_autzer_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_autzer_label,2,1,1,1);
	
	prbtype_label = new QLabel(this);
	prbtype_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	prbtype_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(prbtype_label,3,0,1,1);
	
	d_prbtype_label = new QLabel(this);
	d_prbtype_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_prbtype_label,3,1,1,1);

	prbstatus_label = new QLabel(this);
	prbstatus_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	prbstatus_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(prbstatus_label,4,0,1,1);
	
	d_prbstatus_label = new QLabel(this);
	d_prbstatus_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_prbstatus_label,4,1,1,1);

	machine_type_label = new QLabel(this);
	machine_type_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	machine_type_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(machine_type_label,5,0,1,1);
	
	d_machine_type_label = new QLabel(this);
	d_machine_type_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_machine_type_label,5,1,1,1);
	
	machine_serial_label = new QLabel(this);
	machine_serial_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	machine_serial_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(machine_serial_label,6,0,1,1);
	
	d_machine_serial_label = new QLabel(this);
	d_machine_serial_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_machine_serial_label,6,1,1,1);
	
	fw_version_label = new QLabel(this);
	fw_version_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	fw_version_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(fw_version_label,7,0,1,1);
	
	d_fw_version_label = new QLabel(this);
	d_fw_version_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_fw_version_label,7,1,1,3);

	tempcomptype_label = new QLabel(this);
	tempcomptype_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	tempcomptype_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(tempcomptype_label,0,2,1,1);

	d_tempcomptype_label = new QLabel(this);
	d_tempcomptype_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_tempcomptype_label,0,3,1,1);
	
	volcomp_type_label = new QLabel(this);
	volcomp_type_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	volcomp_type_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(volcomp_type_label,1,2,1,1);
	
	d_volcomp_type_label = new QLabel(this);
	d_volcomp_type_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_volcomp_type_label,1,3,1,1);
	
	volcomp_label = new QLabel(this);
	volcomp_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	volcomp_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(volcomp_label,2,2,1,1);
	
	d_volcomp_label = new QLabel(this);
	d_volcomp_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_volcomp_label,2,3,1,1);
	
	geo_label = new QLabel(this);
	geo_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	geo_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(geo_label,3,2,1,1);
	
	d_geo_label = new QLabel(this);
	d_geo_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_geo_label,3,3,1,1);
	
	mapstat_label = new QLabel(this);
	mapstat_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	mapstat_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(mapstat_label,4,2,1,1);
	
	d_mapstat_label = new QLabel(this);
	d_mapstat_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_mapstat_label,4,3,1,1);
	
	dyncomp_label = new QLabel(this);
	dyncomp_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	dyncomp_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(dyncomp_label,5,2,1,1);
	
	d_dyncomp_label = new QLabel(this);
	d_dyncomp_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_dyncomp_label,5,3,1,1);
	
	x_cte_label = new QLabel(this);
	x_cte_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	x_cte_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(x_cte_label,0,4,1,1);
	
	d_x_cte_label = new QLabel(this);
	d_x_cte_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_x_cte_label,0,5,1,3);
	
	y_cte_label = new QLabel(this);
	y_cte_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	y_cte_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(y_cte_label,1,4,1,1);
	
	d_y_cte_label = new QLabel(this);
	d_y_cte_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_y_cte_label,1,5,1,3);
	
	z_cte_label = new QLabel(this);
	z_cte_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	z_cte_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(z_cte_label,2,4,1,1);
	
	d_z_cte_label = new QLabel(this);
	d_z_cte_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_z_cte_label,2,5,1,3);

	x_temperature_label = new QLabel(this);
	x_temperature_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	x_temperature_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(x_temperature_label,3,4,1,1);
	
	d_x_temperature_label = new QLabel(this);
	d_x_temperature_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_x_temperature_label,3,5,1,3);
	
	y_temperature_label = new QLabel(this);
	y_temperature_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	y_temperature_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(y_temperature_label,4,4,1,1);
	
	d_y_temperature_label = new QLabel(this);
	d_y_temperature_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_y_temperature_label,4,5,1,3);
	
	z_temperature_label = new QLabel(this);
	z_temperature_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	z_temperature_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(z_temperature_label,5,4,1,1);
	
	d_z_temperature_label = new QLabel(this);
	d_z_temperature_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_z_temperature_label,5,5,1,3);
	
	w_temperature_label = new QLabel(this);
	w_temperature_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	w_temperature_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(w_temperature_label,6,4,1,1);
	
	d_w_temperature_label = new QLabel(this);
	d_w_temperature_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_w_temperature_label,6,5,1,3);
	
	part_temperature_label = new QLabel(this);
	part_temperature_label->setAlignment(Qt::AlignTrailing|Qt::AlignRight);
	part_temperature_label->setSizePolicy(size_policy_maximum_preferred);
	status_layout->addWidget(part_temperature_label,7,4,1,1);
	
	d_part_temperature_label = new QLabel(this);
	d_part_temperature_label->setSizePolicy(size_policy_minexp_preferred);
	status_layout->addWidget(d_part_temperature_label,7,5,1,3);
	
	dialog_layout->addLayout(status_layout,5,0,1,5);

	separator_hline2 = new QFrame(this);
	separator_hline2->setFrameShape(QFrame::HLine);
	separator_hline2->setFrameShadow(QFrame::Sunken);
	dialog_layout->addWidget(separator_hline2,6,0,1,5);

	error_history_label = new QLabel(this);
	error_history_label->setFont(title_font);
	dialog_layout->addWidget(error_history_label,7,0,1,5);

	d_controller_log_edit = new QPlainTextEdit(this);
	d_controller_log_edit->setFont(text_font);
	d_controller_log_edit->setLineWrapMode(QPlainTextEdit::NoWrap);
	d_controller_log_edit->setReadOnly(true);
	dialog_layout->addWidget(d_controller_log_edit,8,0,1,5);
	
	communication_log_label = new QLabel(this);
	communication_log_label->setFont(title_font);
	dialog_layout->addWidget(communication_log_label,9,0,1,5);
	
	QFontMetrics						font_metrics(title_font);

	d_communication_log_edit = new QPlainTextEdit(this);
	d_communication_log_edit->setFont(text_font);
	d_communication_log_edit->setLineWrapMode(QPlainTextEdit::NoWrap);
	d_communication_log_edit->setReadOnly(true);
	d_communication_log_edit->setMaximumSize(QSize(16777215, font_metrics.height() * 6));
	dialog_layout->addWidget(d_communication_log_edit,10,0,1,5);
	
	d_relays_label->setMinimumWidth(font_metrics.averageCharWidth() * 12);
	d_tempcomptype_label->setMinimumWidth(font_metrics.averageCharWidth() * 12);
	d_x_cte_label->setMinimumWidth(font_metrics.averageCharWidth() * 12);
	
	logo_label = new QLabel(this);
	logo_label->setMinimumSize(100,40);
	dialog_layout->addWidget(logo_label,11,0,1,2);

	copyright_label = new QLabel(this);
	copyright_label->setFont(copyright_font);
	copyright_label->setAlignment(Qt::AlignBottom|Qt::AlignTrailing|Qt::AlignRight);
	dialog_layout->addWidget(copyright_label,11,2,1,3);
	
	logo_label->setPixmap(QPixmap(QStringLiteral(":/images/logo.png")));
	d_update_button->setIcon(QIcon(QStringLiteral(":/icon_48/update_48.png")));
	d_save_button->setIcon(QIcon(QStringLiteral(":/icon_48/save_48.png")));
	
	d_controller_ip_edit->setText(d_settings->value(QStringLiteral("Controller_IP_Address"), QStringLiteral("100.0.0.1")).toString());
	d_controller_port_edit->setText(d_settings->value(QStringLiteral("Controller_Port"), QStringLiteral("1234")).toString());
	
	d_close_socket_timer = new QTimer(this);
	
	// defaults
	d_active_tcp_socket = 0;
	d_save_button->setEnabled(false);

	this->setWindowTitle(QStringLiteral("DC Controller Status - Version 1.3"));
	
	controller_connection_label->setText(QStringLiteral("DC Controller"));
	controller_ip_label->setText(QStringLiteral("IP Address:"));
	controller_port_label->setText(QStringLiteral("Port:"));
	d_update_button->setToolTip(QStringLiteral("Connect to the controller and get error history data."));
	d_save_button->setToolTip(QStringLiteral("Save the current error data to a file."));
	machine_status_label->setText(QStringLiteral("Current Machine Status"));
	relays_label->setText(QStringLiteral("Relay State:"));
	estop_label->setText(QStringLiteral("Emergency State:"));
	autzer_label->setText(QStringLiteral("Machine Homed:"));
	prbtype_label->setText(QStringLiteral("Probe Type:"));
	prbstatus_label->setText(QStringLiteral("Probe Status:"));
	machine_type_label->setText(QStringLiteral("Machine Type:"));
	machine_serial_label->setText(QStringLiteral("Machine Serial:"));
	fw_version_label->setText(QStringLiteral("FW Version:"));
	tempcomptype_label->setText(QStringLiteral("Temp. Comp. Type:"));
	volcomp_type_label->setText(QStringLiteral("Volcomp Type:"));
	volcomp_label->setText(QStringLiteral("Volcomp:"));
	geo_label->setText(QStringLiteral("GEO State:"));
	mapstat_label->setText(QStringLiteral("Map State:"));
	dyncomp_label->setText(QStringLiteral("Dynamic Comp:"));
	x_cte_label->setText(QStringLiteral("X CTE:"));
	y_cte_label->setText(QStringLiteral("Y CTE:"));
	z_cte_label->setText(QStringLiteral("Z CTE:"));
	x_temperature_label->setText(QStringLiteral("X Temperature:"));
	y_temperature_label->setText(QStringLiteral("Y Temperature:"));
	z_temperature_label->setText(QStringLiteral("Z Temperature:"));
	w_temperature_label->setText(QStringLiteral("W Temperature:"));
	part_temperature_label->setText(QStringLiteral("Part Temperature:"));
	error_history_label->setText(QStringLiteral("Error History"));
	communication_log_label->setText(QStringLiteral("Communication Log"));
	copyright_label->setText(QStringLiteral("Copyright (C) 2025 Select Calibration Incorporated.  www.selectcalibration.ca"));

	connect(d_update_button,&QToolButton::clicked,this,&TDCControllerStatus::Update);
	connect(d_save_button,&QToolButton::clicked,this,&TDCControllerStatus::Save);
	connect(d_close_socket_timer,&QTimer::timeout,this,&TDCControllerStatus::Close_Socket);
}

TDCControllerStatus::~TDCControllerStatus(void)
{
}

void TDCControllerStatus::Update(void)
{
	QTcpSocket							*tcp_socket;
	
	this->Clear_Status_Log();
	this->Clear_Error_Log();
	
	d_save_button->setEnabled(false);

	d_relays_label->setText(QString());
	d_estop_label->setText(QString());
	d_autzer_label->setText(QString());
	d_prbtype_label->setText(QString());
	d_prbstatus_label->setText(QString());
	d_tempcomptype_label->setText(QString());
	d_machine_type_label->setText(QString());
	d_machine_serial_label->setText(QString());
	d_volcomp_type_label->setText(QString());
	d_volcomp_label->setText(QString());
	d_geo_label->setText(QString());
	d_mapstat_label->setText(QString());
	d_fw_version_label->setText(QString());
	d_x_temperature_label->setText(QString());
	d_y_temperature_label->setText(QString());
	d_z_temperature_label->setText(QString());
	d_w_temperature_label->setText(QString());
	d_part_temperature_label->setText(QString());
	
	tcp_socket = new QTcpSocket;
	
	connect(tcp_socket,&QTcpSocket::connected,this,&TDCControllerStatus::Socket_Connected);
	connect(tcp_socket,&QTcpSocket::disconnected,this,&TDCControllerStatus::Socket_Disconnected);
	
#if (QT_VERSION >= QT_VERSION_CHECK(5, 15, 0))
	connect(tcp_socket,static_cast<void (QAbstractSocket::*)(QAbstractSocket::SocketError)>(&QTcpSocket::errorOccurred),this,&TDCControllerStatus::Socket_Error);
#else
	connect(tcp_socket,static_cast<void (QAbstractSocket::*)(QAbstractSocket::SocketError)>(&QTcpSocket::error),this,&TDCControllerStatus::Socket_Error);
#endif
	
	connect(tcp_socket,static_cast<void (QAbstractSocket::*)(QAbstractSocket::SocketState)>(&QTcpSocket::stateChanged),this,&TDCControllerStatus::Socket_State_Changed);
	connect(tcp_socket,&QTcpSocket::readyRead,this,&TDCControllerStatus::Socket_Read);
	
	tcp_socket->connectToHost(QHostAddress(d_controller_ip_edit->text()),d_controller_port_edit->text().toInt());
}

void TDCControllerStatus::Close_Socket(void)
{
	d_close_socket_timer->stop();

	if(d_active_tcp_socket)
	{
		d_active_tcp_socket->close();
		d_active_tcp_socket = 0;
	}
}

void TDCControllerStatus::Save(void)
{
	QString								file_name;
	QFile								file;
	QString								text;
	
	file_name = QFileDialog::getSaveFileName(
											 this,
											 QStringLiteral("Save Controller Status Data"),
											 d_application_path,
											 QStringLiteral("Text Files (*.txt *.dat *.log)"));
	
	if(file_name.length())
	{
		file.setFileName(file_name);
		
		if(file.open(QIODevice::WriteOnly | QIODevice::Text))
		{
			file.write("Current Machine Status:\n----------------------------------------------------------------------------------------\n\n");
			
			text = d_relays_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Relay State:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_estop_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Emergency State:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_autzer_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Machine Homed:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_prbtype_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Probe Type:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_prbstatus_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Probe Status:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_machine_type_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Machine Type:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_machine_serial_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Machine Serial:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_fw_version_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("FW Version:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());

			text = d_tempcomptype_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Temp Comp Type:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());

			text = d_volcomp_type_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Volcomp Type:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_volcomp_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Volcomp:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_geo_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("GEO State:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_mapstat_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Map State:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_dyncomp_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Dynamic Comp:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_x_cte_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("X CTE:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_y_cte_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Y CTE:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_z_cte_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Z CTE:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());

			text = d_x_temperature_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Temperature X:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_y_temperature_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Temperature Y:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_z_temperature_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Temperature Z:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_w_temperature_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Temperature W:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			text = d_part_temperature_label->text();
			file.write(QString("%1 %2\n").arg(QStringLiteral("Temperature Part:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text).toLatin1());
			
			file.write("\nError History:\n----------------------------------------------------------------------------------------\n\n");
			
			file.write(d_controller_log_edit->toPlainText().toLatin1());

			file.close();
		}
	}
}

void TDCControllerStatus::Socket_Connected(void)
{
	QTcpSocket							*tcp_socket;
	static const int					AUTOCLOSE_QUIET_TIME(500);
	
	tcp_socket = reinterpret_cast<QTcpSocket*>(QObject::sender());
	
	if(d_active_tcp_socket)
	{
		disconnect(d_active_tcp_socket,0,0,0);
		d_active_tcp_socket->deleteLater();
		d_active_tcp_socket = 0;
	}
	
	if(tcp_socket)
	{
		d_active_tcp_socket = tcp_socket;
		
		this->Clear_Error_Log();
		this->Clear_Status_Log();
		this->Add_Status_Log_Text(QStringLiteral("INF:  Connection accepted"));

		d_waiting_for_ready = true;
		d_waiting_for_error_history = false;

		this->Initialize_Connection(tcp_socket);
		this->Initialize_Command_Queue();
		
		d_close_socket_timer->start(AUTOCLOSE_QUIET_TIME);
		
		d_save_button->setEnabled(true);
	}
}

void TDCControllerStatus::Socket_Disconnected(void)
{
	this->Add_Status_Log_Text(QStringLiteral("INF:  Connection closed"));
}

void TDCControllerStatus::Socket_Error(
	QAbstractSocket::SocketError		socket_error)
{
	switch(socket_error)
	{
		case QAbstractSocket::ConnectionRefusedError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The connection was refused by the peer (or timed out)."));
			break;
		case QAbstractSocket::RemoteHostClosedError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The remote host closed the connection. Note that the client socket (i.e., this socket) will be closed after the remote close notification has been sent."));
			break;
		case QAbstractSocket::HostNotFoundError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The host address was not found."));
			break;
		case QAbstractSocket::SocketAccessError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The socket operation failed because the application lacked the required privileges."));
			break;
		case QAbstractSocket::SocketResourceError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The local system ran out of resources (e.g., too many sockets)."));
			break;
		case QAbstractSocket::SocketTimeoutError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The socket operation timed out."));
			break;
		case QAbstractSocket::DatagramTooLargeError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The datagram was larger than the operating system's limit (which can be as low as 8192 bytes)."));
			break;
		case QAbstractSocket::NetworkError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  An error occurred with the network (e.g., the network cable was accidentally plugged out)."));
			break;
		case QAbstractSocket::AddressInUseError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The address specified to QAbstractSocket::bind() is already in use and was set to be exclusive."));
			break;
		case QAbstractSocket::SocketAddressNotAvailableError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The address specified to QAbstractSocket::bind() does not belong to the host."));
			break;
		case QAbstractSocket::UnsupportedSocketOperationError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The requested socket operation is not supported by the local operating system (e.g., lack of IPv6 support)."));
			break;
		case QAbstractSocket::ProxyAuthenticationRequiredError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The socket is using a proxy, and the proxy requires authentication."));
			break;
		case QAbstractSocket::SslHandshakeFailedError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The SSL/TLS handshake failed, so the connection was closed (only used in QSslSocket)."));
			break;
		case QAbstractSocket::UnfinishedSocketOperationError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  Used by QAbstractSocketEngine only, The last operation attempted has not finished yet (still in progress in the background)."));
			break;
		case QAbstractSocket::ProxyConnectionRefusedError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  Could not contact the proxy server because the connection to that server was denied."));
			break;
		case QAbstractSocket::ProxyConnectionClosedError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The connection to the proxy server was closed unexpectedly (before the connection to the final peer was established)."));
			break;
		case QAbstractSocket::ProxyConnectionTimeoutError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The connection to the proxy server timed out or the proxy server stopped responding in the authentication phase."));
			break;
		case QAbstractSocket::ProxyNotFoundError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The proxy address set with setProxy() (or the application proxy) was not found."));
			break;
		case QAbstractSocket::ProxyProtocolError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The connection negotiation with the proxy server failed, because the response from the proxy server could not be understood."));
			break;
		case QAbstractSocket::OperationError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  An operation was attempted while the socket was in a state that did not permit it."));
			break;
		case QAbstractSocket::SslInternalError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  The SSL library being used reported an internal error. This is probably the result of a bad installation or misconfiguration of the library."));
			break;
		case QAbstractSocket::SslInvalidUserDataError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  Invalid data (certificate, key, cypher, etc.) was provided and its use resulted in an error in the SSL library."));
			break;
		case QAbstractSocket::TemporaryError:
			this->Add_Status_Log_Text(QStringLiteral("ERR:  A temporary error occurred (e.g., operation would block and socket is non-blocking)."));
			break;
		case QAbstractSocket::UnknownSocketError:
			this->Add_Status_Log_Text(QString("ERR:  %1").arg(socket_error));
			break;
	}
}

void TDCControllerStatus::Socket_State_Changed(
	QAbstractSocket::SocketState		state)
{
	switch(state)
	{
		case QAbstractSocket::UnconnectedState:
			this->Add_Status_Log_Text(QStringLiteral("INF:  Socket Disconnected"));
			break;
			
		case QAbstractSocket::HostLookupState:
			this->Add_Status_Log_Text(QStringLiteral("INF:  Socket Host Lookup."));
			break;
			
		case QAbstractSocket::ConnectingState:
			this->Add_Status_Log_Text(QStringLiteral("INF:  Socket Connecting..."));
			break;
			
		case QAbstractSocket::ConnectedState:
			this->Add_Status_Log_Text(QStringLiteral("INF:  Socket Connected"));
			break;
			
		case QAbstractSocket::BoundState:
			this->Add_Status_Log_Text(QStringLiteral("INF:  Socket Bound"));
			break;
			
		case QAbstractSocket::ClosingState:
			this->Add_Status_Log_Text(QStringLiteral("INF:  Socket Closing..."));
			break;
			
		case QAbstractSocket::ListeningState:
			this->Add_Status_Log_Text(QStringLiteral("INF:  Socket Listening..."));
			break;
	}
}

void TDCControllerStatus::closeEvent(
	QCloseEvent							*event)
{
	d_settings->setValue("Controller_IP_Address", d_controller_ip_edit->text());
	d_settings->setValue("Controller_Port", d_controller_port_edit->text());
	
	event->accept();
}

void TDCControllerStatus::Socket_Read(void)
{
	QTcpSocket							*tcp_socket;
	
	tcp_socket = reinterpret_cast<QTcpSocket*>(QObject::sender());
	
	if(tcp_socket)
	{
		d_rx_data += tcp_socket->readAll();
		this->Process_Rx_Data(tcp_socket);
	}
}

void TDCControllerStatus::Process_Rx_Data(
	QTcpSocket							* const socket)
{
	int									rx_line_index;
	int									index;
	int									indexb;
	int									ival;
	int									cntr;
	QString								keyword;
	QString								arguments;
	QString								line;
	QString								text;
	QString								write_text;
	QString								response;
	QStringList							list;
	bool								state;
	bool								send_next_command(false);
	double								dval;

	do
	{
		rx_line_index = d_rx_data.indexOf('\r');
		
		if(!(rx_line_index < 0))
		{
			line = QString::fromLatin1(d_rx_data.mid(0,rx_line_index)).simplified();
			d_rx_data.remove(0,rx_line_index + 1);
			
			// clean text
			while(line.length() && line[0].toLatin1() < 0x20)
			{
				line = line.mid(1);
			}
			
			if(line.length())
			{
				
#ifdef DEBUG_TX_RX
				qWarning(QString("RX: '%1'").arg(line).toLatin1());
#endif
				
					if(line.startsWith(QStringLiteral("ERROR"),Qt::CaseInsensitive))
					{
						this->Initialize_Connection(socket);
						d_waiting_for_ready = true;
						
						this->Add_Status_Log_Text(QStringLiteral("WAR:  Controller reported error."));
					}
					else
					{
						index = line.indexOf(' ');
						
						if(!(index < 0))
						{
							keyword = line.mid(0,index).toUpper();
							arguments = line.mid(index + 1);
						}
						else
						{
							keyword = line.toUpper();
							arguments = QString();
						}

						if(!d_waiting_for_ready)
						{
							if(d_waiting_for_error_history)
							{
								if(keyword.compare(QStringLiteral("ERRDIAGN")) == 0)
								{
									if(arguments.length())
									{
										if(arguments.startsWith(QStringLiteral("ERRCOD"),Qt::CaseInsensitive))
										{
											text = arguments.mid(6).simplified();	// remove ERRCOD
											this->Add_Error_Log_Text(QString());
											
											++d_display_error_number;
											write_text = QString("%1 %2\n").arg(QStringLiteral("Error Number:"),OUTPUT_FIELD_TITLE_WIDTH).arg(d_display_error_number);
											write_text += QString("%1 %2").arg(QStringLiteral("Error Code:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("SETTIME"),Qt::CaseInsensitive))
										{
											text = arguments.mid(7).simplified();	// remove SETTIME

											write_text = QString("%1 %2").arg(QStringLiteral("Error Time:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("MODULE"),Qt::CaseInsensitive))
										{
											text = arguments.mid(6).simplified();	// remove MODULE

											write_text = QString("%1 %2").arg(QStringLiteral("Module:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("LABEL"),Qt::CaseInsensitive))
										{
											text = arguments.mid(5).simplified();	// remove LABEL

											write_text = QString("%1 %2").arg(QStringLiteral("Error Label:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("LOCATION"),Qt::CaseInsensitive))
										{
											text = arguments.mid(8).simplified();	// remove LOCATION

											write_text = QString("%1 %2").arg(QStringLiteral("Location:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("ERRTYP"),Qt::CaseInsensitive))
										{
											text = arguments.mid(6).simplified();	// remove ERRTYP

											write_text = QString("%1 %2").arg(QStringLiteral("Error Type:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("CURRCOM"),Qt::CaseInsensitive))
										{
											text = arguments.mid(7).simplified();	// remove CURRCOM

											write_text = QString("%1 %2").arg(QStringLiteral("Current Command:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("MANAUTO"),Qt::CaseInsensitive))
										{
											text = arguments.mid(7).simplified();	// remove MANAUTO
											
											ival = text.toInt(&state);
											
											if(state)
											{
												switch(ival)
												{
													case 0:
														text = QStringLiteral("Manual");
														break;
														
													case 1:
														text = QStringLiteral("Auto");
														break;
														
													default:
														text = QString("Auto %1").arg(ival);
														break;
												}
											}
											
											write_text = QString("%1 %2").arg(QStringLiteral("Man / Auto:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("JOGCNC"),Qt::CaseInsensitive))
										{
											text = arguments.mid(6).simplified();	// remove JOGCNC
											
											ival = text.toInt(&state);
											
											if(state)
											{
												switch(ival)
												{
													case 0:
														text = QStringLiteral("Jogbox");
														break;
														
													case 1:
														text = QStringLiteral("CNC");
														break;
														
													default:
														text = QString("Mode %1").arg(ival);
														break;
												}
											}

											write_text = QString("%1 %2").arg(QStringLiteral("Jog / CNC:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("PRBSTAT"),Qt::CaseInsensitive))
										{
											text = arguments.mid(7).simplified();	// remove PRBSTAT
											ival = text.toInt(&state);
											
											if(state)
											{
												switch(ival)
												{
													case 0:
														text = QStringLiteral("Seated");
														break;
														
													case 1:
														text = QStringLiteral("Unseated");
														break;
														
													default:
														text = QString("State %1").arg(ival);
														break;
												}
											}

											write_text = QString("%1 %2").arg(QStringLiteral("Probe Status:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("RESETTIME"),Qt::CaseInsensitive))
										{
											text = arguments.mid(9).simplified();	// remove RESETTIME

											write_text = QString("%1 %2").arg(QStringLiteral("Error Reset Time:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("ERRPOSLOG"),Qt::CaseInsensitive))
										{
											text = arguments.mid(9).simplified();	// remove ERRPOSLOG
											
											write_text = QString("%1 %2").arg(QStringLiteral("Error Pos. Log:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("ERRNUMPOS"),Qt::CaseInsensitive))
										{
											text = arguments.mid(9).simplified();	// remove ERRNUMPOS
											
											write_text = QString("%1 %2").arg(QStringLiteral("Error Num Pos:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("ERRPOS"),Qt::CaseInsensitive))
										{
											text = arguments.mid(6).simplified();	// remove ERRPOS
											
											write_text = QString("%1 %2").arg(QStringLiteral("Error Position:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("ERRVEL"),Qt::CaseInsensitive))
										{
											text = arguments.mid(7).simplified();	// remove ERRVEL
											
											write_text = QString("%1 %2").arg(QStringLiteral("Velocity:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("PRCHNG"),Qt::CaseInsensitive))
										{
											text = arguments.mid(6).simplified();	// remove PRCHNG
											
											write_text = QString("%1 %2").arg(QStringLiteral("Probe Changing:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("INPUTSTR"),Qt::CaseInsensitive))
										{
											text = arguments.mid(8).simplified();	// remove INPUTSTR
											
											write_text = QString("%1 %2").arg(QStringLiteral("Input String:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										else if(arguments.startsWith(QStringLiteral("SAFEBARR"),Qt::CaseInsensitive))
										{
											text = arguments.mid(8).simplified();	// remove SAFEBARR
											
											ival = text.toInt(&state);
											
											if(state)
											{
												switch(ival)
												{
													case 0:
														text = QStringLiteral("Closed or Not Present");
														break;
														
													case 1:
														text = QStringLiteral("Open");
														break;
														
													default:
														text = QString("State %1").arg(ival);
														break;
												}
											}

											write_text = QString("%1 %2").arg(QStringLiteral("Safety Barrier:"),OUTPUT_FIELD_TITLE_WIDTH).arg(text);
										}
										
										this->Add_Error_Log_Text(write_text);
									}
									
									send_next_command = true;
								}
							}
							else
							{
								if(keyword.compare(QStringLiteral("RELAYS")) == 0)
								{
									d_relays_label->setText(arguments);
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("ESTOP")) == 0)
								{
									d_estop_label->setText(arguments);
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("AUTZER")) == 0)
								{
									d_autzer_label->setText(arguments);
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("PRBTYP")) == 0)
								{
									d_prbtype_label->setText(arguments);
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("PRBSTATUS")) == 0)
								{
									d_prbstatus_label->setText(arguments);

									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("MACHINETYPE")) == 0)
								{
									d_machine_type_label->setText(arguments);
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("SERIALNUMBER")) == 0)
								{
									d_machine_serial_label->setText(arguments);
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("TEMPCOMPTYPE")) == 0)
								{
									index = arguments.toInt(&state);
									
									if(state)
									{
										switch(index)
										{
											case 0:
												d_tempcomptype_label->setText(QStringLiteral("None (0)"));
												break;
											case 1:
												d_tempcomptype_label->setText(QStringLiteral("CFW Method (1)"));
												break;
											case 2:
												d_tempcomptype_label->setText(QStringLiteral("Parametric (2)"));
												break;
											default:
												d_tempcomptype_label->setText(QString("%1").arg(index));
												break;
										}
									}
									else
									{
										d_tempcomptype_label->setText(QStringLiteral("<unknown>"));
									}
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("VOLCOMPTYPE")) == 0)
								{
									index = arguments.toInt(&state);
									
									if(state)
									{
										switch(index)
										{
											case 0:
												d_volcomp_type_label->setText(QStringLiteral("None (0)"));
												break;
											case 1:
												d_volcomp_type_label->setText(QStringLiteral("DEA (1)"));
												break;
											case 2:
												d_volcomp_type_label->setText(QStringLiteral("BnS (2)"));
												break;
											default:
												d_volcomp_type_label->setText(QString("%1").arg(index));
												break;
										}
									}
									else
									{
										d_volcomp_type_label->setText(QStringLiteral("<unknown>"));
									}
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("VOLCOMP")) == 0)
								{
									d_volcomp_label->setText(arguments);
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("GEO")) == 0)
								{
									d_geo_label->setText(arguments);
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("MAPSTAT")) == 0)
								{
									d_mapstat_label->setText(arguments);
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("DYNCOMP")) == 0)
								{
									d_dyncomp_label->setText(arguments);
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("X_NCESCL")) == 0)
								{
									list = arguments.split(',');
									
									if(list.size() > 0)
									{
										dval = list[0].toDouble();
										d_x_cte_label->setText(QString("%1 um/m/C").arg(dval * 1000.0,0,'f',1));
									}
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("Y_NCESCL")) == 0)
								{
									list = arguments.split(',');
									
									if(list.size() > 0)
									{
										dval = list[0].toDouble();
										d_y_cte_label->setText(QString("%1 um/m/C").arg(dval * 1000.0,0,'f',1));
									}
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("Z_NCESCL")) == 0)
								{
									list = arguments.split(',');
									
									if(list.size() > 0)
									{
										dval = list[0].toDouble();
										d_z_cte_label->setText(QString("%1 um/m/C").arg(dval * 1000.0,0,'f',1));
									}
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("SCLTMP")) == 0)
								{
									list = arguments.split(',');
									
									if(list.size() > 3)
									{
										dval = list[0].toDouble();
										d_x_temperature_label->setText(QString("%1 C").arg(dval,0,'f',2));

										dval = list[1].toDouble();
										d_y_temperature_label->setText(QString("%1 C").arg(dval,0,'f',2));

										dval = list[2].toDouble();
										d_z_temperature_label->setText(QString("%1 C").arg(dval,0,'f',2));

										dval = list[3].toDouble();
										d_w_temperature_label->setText(QString("%1 C").arg(dval,0,'f',2));
									}

									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("WKPPAR")) == 0)
								{
									list = arguments.split(',');
									
									if(list.size() > 0)
									{
										dval = list[0].toDouble();
										d_part_temperature_label->setText(QString("%1 C").arg(dval,0,'f',2));
									}
									
									send_next_command = true;
								}
								else if(keyword.compare(QStringLiteral("ERRHISTNUM")) == 0)
								{
									index = arguments.toInt(&state);
									
									if(state)
									{
										if(index)
										{
											this->Add_Status_Log_Text(QString("INF:  Controller error history log entry count: %1").arg(index));
										}
										else
										{
											this->Add_Status_Log_Text(QStringLiteral("INF:  Controller error history log is empty."));
										}
										
										d_display_error_number = 0;
										
										for(cntr = 0;cntr < index;++cntr)
										{
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,ERRCOD\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,SETTIME\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,RESTIME\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,INPUTSTR\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,LABEL\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,ERRTYP\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,MODULE\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,LOCATION\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,ERRPOS\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,ERRPOSLOG\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,ERRNUMPOS\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,ERRVEL\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,PRCHNG\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,CURRCOM\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,MANAUTO\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,JOGCNC\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,PRBSTAT\r\n").arg(cntr+1));
											d_command_queue.push_back(QString("SHOW ERRDIAGN, %1,SAFEBARR\r\n").arg(cntr+1));
										}
									}
									
									send_next_command = true;
								}
								else if(line.contains(QStringLiteral("FDC"),Qt::CaseInsensitive) &&
										line.contains(QStringLiteral("DATE:"),Qt::CaseInsensitive))
								{
									index = line.indexOf("FDC",Qt::CaseInsensitive);
									indexb = line.indexOf("DATE:",Qt::CaseInsensitive);
									
									response = line.mid(index + 3,indexb - index - 3);
									
									d_fw_version_label->setText(response.simplified());
									
									send_next_command = true;
								}
							}
							
							if(send_next_command && d_command_queue.size())
							{
								text = d_command_queue[0];
								d_command_queue.erase(d_command_queue.begin());
								
								if(text.startsWith(QStringLiteral("SHOW ERRDIAGN"),Qt::CaseInsensitive))
								{
									d_waiting_for_error_history = true;
								}
#ifdef DEBUG_TX_RX
								qWarning(QString("TX: %1").arg(text).toLatin1());
#endif
								socket->write(text.toLatin1());
							}
						}
						else if(keyword.compare(QStringLiteral("READY")) == 0)
						{
							d_waiting_for_ready = false;
							d_save_button->setEnabled(true);
							
							QThread::msleep(20);
							
							if(d_command_queue.size())
							{
								text = d_command_queue[0];
								d_command_queue.erase(d_command_queue.begin());
								
								if(text.compare(QStringLiteral("SHOW ERRDIAGN")) == 0)
								{
									d_waiting_for_error_history = true;
								}

#ifdef DEBUG_TX_RX
								qWarning(QString("TX: %1").arg(text).toLatin1());
#endif
								socket->write(text.toLatin1());
							}
						}
					}
			}
		}
		
	}while(!(rx_line_index < 0));
	
	d_close_socket_timer->start();
}

void TDCControllerStatus::Clear_Error_Log(void)
{
	d_controller_log_edit->clear();
}

void TDCControllerStatus::Add_Error_Log_Text(
	const QString &text)
{
	d_controller_log_edit->moveCursor(QTextCursor::End);
	d_controller_log_edit->appendPlainText(text);
}

void TDCControllerStatus::Clear_Status_Log(void)
{
	d_communication_log_edit->clear();
}

void TDCControllerStatus::Add_Status_Log_Text(
	const QString &text)
{
	d_communication_log_edit->moveCursor(QTextCursor::End);
	d_communication_log_edit->appendPlainText(text);
}

void TDCControllerStatus::Initialize_Connection(
	QTcpSocket							* const socket)
{
	QByteArray							reset_sequence;
	QByteArray							ctrl_cb;
	static const int					SLEEP_TIME(50);

	reset_sequence.resize(2);
	ctrl_cb.resize(3);
	
	reset_sequence[0] = static_cast<char>(0x0b);
	reset_sequence[1] = static_cast<char>(0x0a);
	
	ctrl_cb[0] = static_cast<char>(0x0e);
	ctrl_cb[1] = static_cast<char>(0x03);
	ctrl_cb[2] = static_cast<char>(0x02);

	socket->write(reset_sequence);
	QThread::msleep(SLEEP_TIME);
	
	socket->write(ctrl_cb);
	QThread::msleep(SLEEP_TIME);	
}

void TDCControllerStatus::Initialize_Command_Queue(void)
{
	d_command_queue.clear();
	
	d_command_queue.push_back(QStringLiteral("SHOW RELAYS\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW ESTOP\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW AUTZER\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW PRBTYP\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW PRBSTATUS\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW MACHINETYPE\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW SERIALNUMBER\r\n"));
	d_command_queue.push_back(QStringLiteral("VERSION\r\n"));
	
	d_command_queue.push_back(QStringLiteral("SHOW TEMPCOMPTYPE\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW VOLCOMPTYPE\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW VOLCOMP\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW GEO\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW MAPSTAT\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW DYNCOMP\r\n"));
	
	d_command_queue.push_back(QStringLiteral("SHOW X_NCESCL\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW Y_NCESCL\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW Z_NCESCL\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW SCLTMP\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW WKPPAR\r\n"));
	d_command_queue.push_back(QStringLiteral("SHOW ERRHISTNUM\r\n"));

}





