/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <assert.h>

#include "vector2.h"

TVector2::TVector2(void)
:x(0),
 y(0)
{
}

TVector2::TVector2(
    const TVector2                  	&lhs)
{
	x = lhs.x;
	y = lhs.y;
}

TVector2::TVector2(
    const double                  		&X,
    const double                  		&Y)
:x(X),
 y(Y)
{
}

TVector2::~TVector2(void)
{

}

double TVector2::operator()(
	const int							Offset) const
{
	switch(Offset)
	{
		case 0:
			return x;

		case 1:
			return y;
	}

	assert(false);

	return 0;
}

double& TVector2::operator()(
	const int							Offset)
{
	switch(Offset)
	{
		case 0:
			return x;

		case 1:
			return y;
	}

	assert(false);

	return x;
}

TVector2& TVector2::Normal(void)
{
     double                        		Length;

     Length = sqrt(x*x + y*y);

     assert(Length > std::numeric_limits<double>::epsilon());

     x/= Length;
     y/= Length;

     return *this;
}

bool TVector2::Is_Point_In_Triangle(
	const TVector2						&p,
	const TVector2						&tp1,
	const TVector2						&tp2,
	const TVector2						&tp3)
{
	double								u,v,w;
	
	// barycentric method
	u = tp1.y * tp3.x - tp1.x * tp3.y + (tp3.y - tp1.y) * p.x + (tp1.x - tp3.x) * p.y;
	v = tp1.x * tp2.y - tp1.y * tp2.x + (tp1.y - tp2.y) * p.x + (tp2.x - tp1.x) * p.y;
	
	if ((u < 0) == (v < 0))
	{
		w = -tp2.y * tp3.x + tp1.y * (tp3.x - tp2.x) + tp1.x * (tp2.y - tp3.y) + tp2.x * tp3.y;
		
		if (w < 0.0)
		{
			u = -u;
			v = -v;
			w = -w;
		}
		
		return ((u > 0) &&
				(v > 0) &&
				(!((u + v) > w)));
	}
	
	return false;
}

double operator^(
     const TVector2                  	&a,
     const TVector2                  	&b)
{
     return a.i*b.i + a.j*b.j;
}

TVector2 operator*(
     const TVector2                  	&a,
     const double                  	&d)
{
     TVector2 v(a);
     return TVector2(a)*=d;
}

TVector2 operator*(
     const double                  	&d,
     const TVector2                  	&a)
{
     return TVector2(a)*=d;
}

TVector2 operator+(
     const TVector2                  	&a,
     const TVector2                  	&b)
{
     return TVector2(a)+=b;
}

TVector2 operator+(
     const TVector2                  	&a,
     const double                  	&d)
{
     return TVector2(a)+=d;
}

TVector2 operator+(
     const double                  	&d,
     const TVector2                  	&a)
{
     return TVector2(a)+=d;
}

TVector2 operator-(
     const TVector2                  	&a,
     const TVector2                  	&b)
{
     return TVector2(a)-=b;
}

TVector2 operator-(
     const TVector2                  	&a,
     const double                  	&d)
{
     return TVector2(a)-=d;
}

TVector2 operator-(
     const double                  	&d,
     const TVector2                  	&a)
{
     return TVector2(a)-=d;
}

TVector2 operator/(
	const TVector2					&v,
	const double					&d)
{
    return TVector2(v)/=d;
}
