/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#ifndef VECTOR2HEADERFILE
#define VECTOR2HEADERFILE

#include <limits>
#include <math.h>
#include <assert.h>

class TVector2
{
public:
// CREATORS
	TVector2(void);
	TVector2(const TVector2&);
	TVector2(const double &X,const double &Y);
	~TVector2(void);

// ACCESSORS
	union
	{
										double x;
										double i;
	};
	union
	{
										double y;
										double j;
	};

	double 			Length(void) const {return sqrt(x*x + y*y);}
	double 			SquaredLength(void) const {return x*x + y*y;}

	// indexed value.  0 = x, 1 = y
	double 			operator()(const int Offset) const;

// MANIPULATORS
	double& 		operator()(const int Offset);

	void 			Set(const double &X,const double &Y);
	void 			Zero(void) {x=0;y=0;};
	TVector2& 		Normal(void);

	TVector2& 		operator=(const TVector2 &lhs);
	TVector2& 		operator+=(const TVector2 &lhs);
	TVector2& 		operator-=(const TVector2 &lhs);

	TVector2& 		operator*=(const double &d);
	TVector2& 		operator/=(const double &d);
	TVector2& 		operator+=(const double &d);
	TVector2& 		operator-=(const double &d);

	// static namespace functions
	static bool Is_Point_In_Triangle(const TVector2 &p,const TVector2 &tp1,const TVector2 &tp2,const TVector2 &tp3);	// points on verticies are not inside
	
private:
// NOT IMPLEMENTED
};

// Inline member functions

inline void TVector2::Set(
    const double                  		&X,
    const double                  		&Y)
{
	x=X;
	y=Y;
}

inline TVector2& TVector2::operator=(
    const TVector2                  	&lhs)
{
	if(&lhs != this)
	{
		x = lhs.x;
		y = lhs.y;
	}

	return *this;
}

inline TVector2& TVector2::operator+=(
    const TVector2                  	&lhs)
{
	if(&lhs != this)
	{
		x += lhs.x;
		y += lhs.y;
	}

	return *this;
}

inline TVector2& TVector2::operator-=(
    const TVector2                  	&lhs)
{
	if(&lhs != this)
	{
		x -= lhs.x;
		y -= lhs.y;
	}

	return *this;
}

inline TVector2& TVector2::operator*=(
    const double                  		&d)
{
	x *= d;
	y *= d;

	return *this;
}

inline TVector2& TVector2::operator/=(
    const double                  		&d)
{
    assert(d > std::numeric_limits<double>::epsilon());

	x /= d;
	y /= d;

	return *this;
}

inline TVector2& TVector2::operator+=(
    const double                  		&d)
{
	x += d;
	y += d;

	return *this;
}

inline TVector2& TVector2::operator-=(
    const double                  		&d)
{
	x -= d;
	y -= d;

	return *this;
}

// Operators
double 			operator^(const TVector2&,const TVector2&);
TVector2 		operator*(const TVector2&,const double&);
TVector2 		operator*(const double&,const TVector2&);
TVector2 		operator+(const TVector2&,const TVector2&);
TVector2 		operator+(const TVector2&,const double&);
TVector2 		operator+(const double&,const TVector2&);
TVector2 		operator-(const TVector2&,const TVector2&);
TVector2 		operator-(const TVector2&,const double&);
TVector2 		operator-(const double&,const TVector2&);
TVector2 		operator/(const TVector2&,const double&);

#endif
