/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QPaintEvent>
#include <QResizeEvent>
#include <QPainter>
#include <QPen>
#include <QBrush>
#include <QColor>
#include <QPoint>
#include <QFontMetrics>
#include <cmath>
#include <assert.h>

#include "deflectionmodel.h"
#include "genericdeflectionmodel.h"
#include "renishawdeflectionmodel.h"
#include "lkdeflectionmodel.h"

#include "displaywidget.h"

TDisplayWidget::TDisplayWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	// defaults
	d_minimum_size = QSize(250,250);
	d_deflection_model = 0;
	
	d_error_multiplier = 1.0;
	d_update_viewport = false;
}

TDisplayWidget::~TDisplayWidget(void)
{
	if(d_deflection_model)
	{
		delete d_deflection_model;
		d_deflection_model = 0;
	}
}

double TDisplayWidget::Model_Parameter1(void) const
{
	if(d_deflection_model)
	{
		return d_deflection_model->Parameter1();
	}
	
	return 0.0;
}

double TDisplayWidget::Model_Parameter2(void) const
{
	if(d_deflection_model)
	{
		return d_deflection_model->Parameter2();
	}
	
	return 0.0;
}

double TDisplayWidget::Model_Parameter3(void) const
{
	if(d_deflection_model)
	{
		return d_deflection_model->Parameter3();
	}
	
	return 0.0;
}

double TDisplayWidget::Model_Parameter4(void) const
{
	if(d_deflection_model)
	{
		return d_deflection_model->Parameter4();
	}
	
	return 0.0;
}


TVector3 TDisplayWidget::Get_Gauge_Point(
	const TVector3 						&pnt) const
{
	TVector3							correction;
	
	if(d_deflection_model)
	{
		correction = d_deflection_model->Calculate_Correction(pnt);
	}
	
	return (pnt - correction);
}

void TDisplayWidget::Set_Range_Y(
	const double						&d)
{
	if(d_deflection_model)
	{
		d_deflection_model->Set_Range_Y(d);
	}
	
	d_update_viewport = true;
	this->update();
}

void TDisplayWidget::Set_Range_Z(
	const double						&d)
{
	if(d_deflection_model)
	{
		d_deflection_model->Set_Range_Z(d);
	}
	
	d_update_viewport = true;
	this->update();
}

void TDisplayWidget::Set_Range(
	const double						&range_y,
	const double						&range_z)
{
	if(d_deflection_model)
	{
		d_deflection_model->Set_Range_Y(range_y);
		d_deflection_model->Set_Range_Z(range_z);
	}
	
	d_update_viewport = true;
	this->update();
}

void TDisplayWidget::Set_Model_Parameter1(
	const double						&param1)
{
	if(d_deflection_model)
	{
		d_deflection_model->Set_Parameter1(param1);
	}
	
	this->update();
}

void TDisplayWidget::Set_Model_Parameter2(
	const double						&param2)
{
	if(d_deflection_model)
	{
		d_deflection_model->Set_Parameter2(param2);
	}
	
	this->update();
}

void TDisplayWidget::Set_Model_Parameter3(
	const double						&param3)
{
	if(d_deflection_model)
	{
		d_deflection_model->Set_Parameter3(param3);
	}
	
	this->update();
}

void TDisplayWidget::Set_Model_Parameter4(
	const double						&param4)
{
	if(d_deflection_model)
	{
		d_deflection_model->Set_Parameter4(param4);
	}
	
	this->update();
}

void TDisplayWidget::Set_Model_Parameters(
	const double						&param1,
	const double						&param2,
	const double						&param3,
	const double						&param4)
{
	if(d_deflection_model)
	{
		d_deflection_model->Set_Parameter1(param1);
		d_deflection_model->Set_Parameter2(param2);
		d_deflection_model->Set_Parameter3(param3);
		d_deflection_model->Set_Parameter4(param4);
	}
	
	this->update();
}

void TDisplayWidget::Set_Exaggeration(
	const int 							i)
{
	if(i < 1)
	{
		d_error_multiplier = 1.0;
	}
	else
	{
		d_error_multiplier = static_cast<double>(i);
	}
	
	this->update();
}

void TDisplayWidget::Set_Deflection_Model(
	const TDisplayWidget::TDeflectionModelType	&model)
{
	if(d_deflection_model)
	{
		delete d_deflection_model;
		d_deflection_model = 0;
	}
	
	switch(model)
	{
		case TDisplayWidget::NO_MODEL:
			break;
			
		case TDisplayWidget::GENERIC_MODEL:
			d_deflection_model = new TGenericDeflectionModel;
			break;
			
		case TDisplayWidget::RENISHAW_MODEL:
			d_deflection_model = new TRenishawDeflectionModel;
			break;
			
		case TDisplayWidget::LK_MODEL:
			d_deflection_model = new TLKDeflectionModel;
			break;
	}
}

void TDisplayWidget::Clear_Length_Measurements(void)
{
	d_length_measurements.clear();
	
	d_update_viewport = true;
	this->update();
}

void TDisplayWidget::Add_Length_Measurement(
	const int							id,
	const TVector3						&p1,
	const TVector3						&p2)
{
	TDisplayWidget::TLengthMeasurement	length_measurement;
	std::vector<TDisplayWidget::TLengthMeasurement>::iterator iter;
	
	length_measurement.p1 = p1;
	length_measurement.p2 = p2;
	length_measurement.id = id;
	length_measurement.selected = false;

	for(iter = d_length_measurements.begin();iter != d_length_measurements.end();++iter)
	{
		if((*iter).id == id)
		{
			return;
		}
	}
	
	d_length_measurements.push_back(length_measurement);
	
	d_update_viewport = true;
	this->update();
}

void TDisplayWidget::Update_Length_Measurement(
	const int							id,
	const TVector3						&p1,
	const TVector3						&p2)
{
	TDisplayWidget::TLengthMeasurement	length_measurement;
	std::vector<TDisplayWidget::TLengthMeasurement>::iterator iter;
	
	length_measurement.p1 = p1;
	length_measurement.p2 = p2;
	length_measurement.id = id;
	length_measurement.selected = false;

	for(iter = d_length_measurements.begin();iter != d_length_measurements.end();++iter)
	{
		if((*iter).id == id)
		{
			(*iter) = length_measurement;
			
			d_update_viewport = true;
			this->update();

			return;
		}
	}
}

void TDisplayWidget::Select_Length_Measurement(
	const int							id)
{
	std::vector<TDisplayWidget::TLengthMeasurement>::iterator iter;

	for(iter = d_length_measurements.begin();iter != d_length_measurements.end();++iter)
	{
		if((*iter).id == id)
		{
			(*iter).selected = true;
		}
		else
		{
			(*iter).selected = false;
		}
	}
	
	d_update_viewport = true;
	this->update();
}

void TDisplayWidget::Remove_Length_Measurement(
	const int							id)
{
	std::vector<TDisplayWidget::TLengthMeasurement>::iterator iter;

	for(iter = d_length_measurements.begin();iter != d_length_measurements.end();++iter)
	{
		if((*iter).id == id)
		{
			d_length_measurements.erase(iter);
			
			d_update_viewport = true;
			this->update();
			
			return;
		}
	}
}

void TDisplayWidget::paintEvent(
	QPaintEvent							*event)
{
	QPainter							painter(this);
	QPen								grey_pen(QColor(164,164,164));
	QPen								red_pen(QColor(196,0,0));
	QPen								black_pen(QColor(0,0,0));
	QPen								arrow_pen(QColor(0,0,0));
	QPen								gauge_pen(QColor(164,164,196));
	QBrush								gauge_brush(QColor(180,180,212));
	QBrush								gauge_brush_selected(QColor(120,120,212));
	QRect								widget_rect;
	QRect								label_bounding_rect;
	QRect								label_rect;
	QRect								rect;
	QPoint								p1,p2,p3,p4;
	TVector3							min_volume,max_volume;
	TVector3							pt;
	TVector3							nominal_p1,nominal_p2;
	TVector3							correction_p1,correction_p2;
	TVector3							length_p1,length_p2;
	std::vector<TDisplayWidget::TLengthMeasurement>::const_iterator iter;
	bool								init;
	int									cntr_y,cntr_z;
	int									xi,yi;
	int									ival;
	double								grid_incr_y,grid_incr_z;
	double								length_incr_y,length_incr_z;
	double								pos_y,pos_z;
	double								length;
	double								range_y,range_z;
	static const int					GRID_LINE_COUNT(25);
	static const int					LENGTH_LINE_COUNT(5);
	static const int					LENGTH_LINE_MARKER(20);
	static const int					LENGTH_LINE_MARKER_SPACER(5);
	static const int					LENGTH_AXIS_LINE(60);
	static const int					ARROW_LENGTH(12);
	static const int					ARROW_WIDTH(3);
	static const int					GAUGE_WIDTH(16);
	
	event->accept();
	
	if(!d_deflection_model)
	{
		return;
	}
	
	arrow_pen.setWidth(2);

	widget_rect = this->rect();
	label_bounding_rect = painter.boundingRect(0,0,0,0,0,QStringLiteral("__999999.99j"));
	
	range_y = d_deflection_model->Range_Y();
	range_z = d_deflection_model->Range_Z();

	if(d_update_viewport)
	{
		if(widget_rect.width() < d_minimum_size.width() || widget_rect.height() < d_minimum_size.height())
		{
			return;
		}
		
		d_grid_rect = widget_rect;
		
		// reduce grid to fit label rect
		ival = label_bounding_rect.width() + LENGTH_LINE_MARKER + LENGTH_LINE_MARKER_SPACER;
		d_grid_rect.adjust(ival,ival,-ival,-ival);
		
		// make grid square
		ival = d_grid_rect.width() - d_grid_rect.height();
		
		if(ival > 0)
		{
			d_grid_rect.adjust(ival/2,0,-ival/2,0);
		}
		else if(ival < 0)
		{
			d_grid_rect.adjust(0,ival/2,0,-ival/2);
		}
		
		min_volume = this->Length_Volume_Minimum();
		max_volume = this->Length_Volume_Maximum();
		
		if(min_volume.y > 0) min_volume.y = 0.0;
		if(min_volume.z > 0) min_volume.z = 0.0;
		
		if(max_volume.y < range_y) max_volume.y = range_y;
		if(max_volume.z < range_z) max_volume.z = range_z;
		
		this->Update_Viewport(min_volume,max_volume);
		
		d_update_viewport = false;
	}
	
	grid_incr_y = range_y / static_cast<double>(GRID_LINE_COUNT - 1);
	grid_incr_z = range_z / static_cast<double>(GRID_LINE_COUNT - 1);
	
	length_incr_y = range_y / static_cast<double>(LENGTH_LINE_COUNT - 1);
	length_incr_z = range_z / static_cast<double>(LENGTH_LINE_COUNT - 1);
	
	// data grid
	
	// y lines
	pos_z = 0.0;
	for(cntr_z = 0;cntr_z < GRID_LINE_COUNT;++cntr_z)
	{
		pos_y = 0.0;
		init = true;
		for(cntr_y = 0;cntr_y < GRID_LINE_COUNT;++cntr_y)
		{
			d_viewport_mat.Position_To_Viewport(pos_y,pos_z,&xi,&yi);
			p1 = QPoint(xi,yi);
			
			pt = this->Calculate_Correction(pos_y,pos_z);
			pt *= d_error_multiplier;
			d_viewport_mat.Position_To_Viewport(pos_y + pt.y,pos_z + pt.z,&xi,&yi);
			p3 = QPoint(xi,yi);

			if(init)
			{
				init = false;
			}
			else
			{
				painter.setPen(grey_pen);
				painter.drawLine(p1,p2);
				
				painter.setPen(red_pen);
				painter.drawLine(p3,p4);
			}

			p2 = p1;
			p4 = p3;
			pos_y += grid_incr_y;
		}
		
		pos_z += grid_incr_z;
	}

	// z lines
	pos_y = 0.0;
	for(cntr_y = 0;cntr_y < GRID_LINE_COUNT;++cntr_y)
	{
		pos_z = 0.0;
		init = true;
		for(cntr_z = 0;cntr_z < GRID_LINE_COUNT;++cntr_z)
		{
			d_viewport_mat.Position_To_Viewport(pos_y,pos_z,&xi,&yi);
			p1 = QPoint(xi,yi);
			
			pt = this->Calculate_Correction(pos_y,pos_z);
			pt *= d_error_multiplier;
			d_viewport_mat.Position_To_Viewport(pos_y + pt.y,pos_z + pt.z,&xi,&yi);
			p3 = QPoint(xi,yi);

			if(init)
			{
				init = false;
			}
			else
			{
				painter.setPen(grey_pen);
				painter.drawLine(p1,p2);
				
				painter.setPen(red_pen);
				painter.drawLine(p3,p4);
			}

			p2 = p1;
			p4 = p3;
			pos_z += grid_incr_z;
		}
		
		pos_y += grid_incr_y;
	}
	
	// length measurements
	
	// y lengths
	pos_z = 0.0;
	init = true;
	
	for(cntr_z = 0;cntr_z < LENGTH_LINE_COUNT;++cntr_z)
	{
		nominal_p1 = TVector3(0.0,0.0,pos_z);
		nominal_p2 = TVector3(0.0,range_y,pos_z);
		
		correction_p1 = this->Calculate_Correction(nominal_p1.y,nominal_p1.z);
		correction_p2 = this->Calculate_Correction(nominal_p2.y,nominal_p2.z);
		
		length_p1 = nominal_p1 + correction_p1;
		length_p2 = nominal_p2 + correction_p2;
		
		correction_p1 *= d_error_multiplier;
		correction_p2 *= d_error_multiplier;
		
		painter.setPen(arrow_pen);
		
		d_viewport_mat.Position_To_Viewport(nominal_p2.y + correction_p2.y,nominal_p2.z + correction_p2.z,&xi,&yi);
		p1 = QPoint(xi,yi);
		
		d_viewport_mat.Position_To_Viewport(nominal_p1.y + correction_p1.y,nominal_p1.z + correction_p1.z,&xi,&yi);
		p2 = QPoint(xi,yi);
		
		this->Draw_Arrow(&painter,p1,p2,ARROW_LENGTH,ARROW_WIDTH);

		if(init)
		{
			label_rect = widget_rect;
			label_rect.setRight(xi - LENGTH_LINE_MARKER - LENGTH_LINE_MARKER_SPACER);

			init = false;
		}
		
		ival = label_bounding_rect.height() / 2;
		
		label_rect.setTop(yi - ival);
		label_rect.setBottom(yi + ival);
		
		pt = length_p2 - length_p1;
		length = pt.Length();
		
		painter.setPen(black_pen);
		painter.drawText(label_rect,Qt::AlignRight | Qt::AlignTrailing |Qt::AlignVCenter,QString("%1").arg(length,0,'f',3));
		
		pos_z += length_incr_z;
	}

	// z lengths
	pos_y = 0.0;
	init = true;
	
	for(cntr_y = 0;cntr_y < LENGTH_LINE_COUNT;++cntr_y)
	{
		nominal_p1 = TVector3(0.0,pos_y,0.0);
		nominal_p2 = TVector3(0.0,pos_y,range_z);
		
		correction_p1 = this->Calculate_Correction(nominal_p1.y,nominal_p1.z);
		correction_p2 = this->Calculate_Correction(nominal_p2.y,nominal_p2.z);
		
		length_p1 = nominal_p1 + correction_p1;
		length_p2 = nominal_p2 + correction_p2;
		
		correction_p1 *= d_error_multiplier;
		correction_p2 *= d_error_multiplier;
		
		painter.setPen(arrow_pen);
		
		d_viewport_mat.Position_To_Viewport(nominal_p2.y + correction_p2.y,nominal_p2.z + correction_p2.z,&xi,&yi);
		p1 = QPoint(xi,yi);
		
		d_viewport_mat.Position_To_Viewport(nominal_p1.y + correction_p1.y,nominal_p1.z + correction_p1.z,&xi,&yi);
		p2 = QPoint(xi,yi);

		this->Draw_Arrow(&painter,p1,p2,ARROW_LENGTH,ARROW_WIDTH);

		if(init)
		{
			label_rect = widget_rect;
			label_rect.setTop(yi + LENGTH_LINE_MARKER + LENGTH_LINE_MARKER_SPACER);
			
			init = false;
		}
		
		ival = label_bounding_rect.height() / 2;
		
		label_rect.setLeft(xi - ival);
		label_rect.setRight(xi + ival);
		
		pt = length_p2 - length_p1;
		length = pt.Length();
		
		painter.save();
		painter.translate(label_rect.center());
		painter.rotate(-90);
		
		rect = label_rect;
		
		rect.setWidth(label_rect.height());
		rect.setHeight(label_rect.width());
		rect.moveCenter(QPoint(0,0));
		
		painter.setPen(black_pen);
		painter.drawText(rect,Qt::AlignRight | Qt::AlignTrailing | Qt::AlignVCenter,QString("%1").arg(length,0,'f',3));
		
		painter.restore();

		pos_y += length_incr_y;
	}
	
	// measured lengths
	painter.setPen(gauge_pen);
	
	for(iter = d_length_measurements.begin();iter != d_length_measurements.end();++iter)
	{
		correction_p1 = this->Calculate_Correction((*iter).p1.y,(*iter).p1.z);
		correction_p2 = this->Calculate_Correction((*iter).p2.y,(*iter).p2.z);
		
		correction_p1 *= d_error_multiplier;
		correction_p2 *= d_error_multiplier;

		d_viewport_mat.Position_To_Viewport((*iter).p1.y + correction_p1.y,(*iter).p1.z + correction_p1.z,&xi,&yi);
		p1 = QPoint(xi,yi);
		
		d_viewport_mat.Position_To_Viewport((*iter).p2.y + correction_p2.y,(*iter).p2.z + correction_p2.z,&xi,&yi);
		p2 = QPoint(xi,yi);
		
		if((*iter).selected)
		{
			painter.setBrush(gauge_brush_selected);
		}
		else
		{
			painter.setBrush(gauge_brush);
		}
		
		this->Draw_Gauge(&painter,p1,p2,QString("%1").arg((*iter).id),GAUGE_WIDTH);
	}
	
	// y axis line
	
	ival = label_bounding_rect.height() / 2;

	p1 = widget_rect.bottomLeft();
	p1 += QPoint(ival,-ival);
	
	p2 = p1;
	p2 += QPoint(LENGTH_AXIS_LINE,0);
	
	painter.setPen(arrow_pen);
	this->Draw_Arrow(&painter,p2,p1,ARROW_LENGTH,ARROW_WIDTH,true);

	ival = label_bounding_rect.height() / 2;
	
	label_rect.setTop(p2.y() - ival);
	label_rect.setBottom(p2.y() + ival);
	label_rect.setLeft(p2.x() + LENGTH_LINE_MARKER_SPACER);
	label_rect.setRight(widget_rect.right());
	
	painter.setPen(black_pen);
	painter.drawText(label_rect,Qt::AlignLeft| Qt::AlignVCenter,QStringLiteral("Y Axis"));
	
	label_rect.setLeft(widget_rect.left());
	painter.drawText(label_rect,Qt::AlignHCenter| Qt::AlignVCenter,QStringLiteral("Z Lengths mm"));

	// z axis line

	ival = label_bounding_rect.height() / 2;

	p1 = widget_rect.bottomLeft();
	p1 += QPoint(ival,-ival);
	
	p2 = p1;
	p2 += QPoint(0,-LENGTH_AXIS_LINE);
	
	painter.setPen(arrow_pen);
	this->Draw_Arrow(&painter,p2,p1,ARROW_LENGTH,ARROW_WIDTH,true);

	label_rect.setTop(widget_rect.top());
	label_rect.setBottom(p2.y() - LENGTH_LINE_MARKER_SPACER);
	label_rect.setLeft(widget_rect.left());
	label_rect.setRight(label_bounding_rect.height());

	painter.save();
	painter.translate(label_rect.center());
	painter.rotate(-90);
	
	rect = label_rect;
	
	rect.setWidth(label_rect.height());
	rect.setHeight(label_rect.width());
	rect.moveCenter(QPoint(0,0));
	
	painter.setPen(black_pen);
	painter.drawText(rect,Qt::AlignLeft| Qt::AlignVCenter,QStringLiteral("Z Axis"));
	
	label_rect.setBottom(widget_rect.bottom());
	
	rect = label_rect;
	
	rect.setWidth(label_rect.height());
	rect.setHeight(label_rect.width());
	rect.moveCenter(QPoint(0,0));

	painter.drawText(rect,Qt::AlignHCenter| Qt::AlignVCenter,QStringLiteral("Y Lengths mm"));

	painter.restore();
}

void TDisplayWidget::resizeEvent(
	QResizeEvent						*event)
{
	QWidget::resizeEvent(event);

	d_update_viewport = true;
}


TVector3 TDisplayWidget::Length_Volume_Minimum(void) const
{
	TVector3							pt;
	bool								init(true);
	std::vector<TDisplayWidget::TLengthMeasurement>::const_iterator iter;
	
	for(iter = d_length_measurements.begin();iter != d_length_measurements.end();++iter)
	{
		if(init)
		{
			init = false;
			
			pt = (*iter).p1;
			
			if((*iter).p2.y < pt.y) pt.y = (*iter).p2.y;
			if((*iter).p2.z < pt.z) pt.z = (*iter).p2.z;
		}
		else
		{
			if((*iter).p1.y < pt.y) pt.y = (*iter).p1.y;
			if((*iter).p1.z < pt.z) pt.z = (*iter).p1.z;
			
			if((*iter).p2.y < pt.y) pt.y = (*iter).p2.y;
			if((*iter).p2.z < pt.z) pt.z = (*iter).p2.z;
		}
	}
	
	return pt;
}

TVector3 TDisplayWidget::Length_Volume_Maximum(void) const
{
	TVector3							pt;
	bool								init(true);
	std::vector<TDisplayWidget::TLengthMeasurement>::const_iterator iter;
	
	for(iter = d_length_measurements.begin();iter != d_length_measurements.end();++iter)
	{
		if(init)
		{
			init = false;
			
			pt = (*iter).p1;
			
			if((*iter).p2.y > pt.y) pt.y = (*iter).p2.y;
			if((*iter).p2.z > pt.z) pt.z = (*iter).p2.z;
		}
		else
		{
			if((*iter).p1.y > pt.y) pt.y = (*iter).p1.y;
			if((*iter).p1.z > pt.z) pt.z = (*iter).p1.z;
			
			if((*iter).p2.y > pt.y) pt.y = (*iter).p2.y;
			if((*iter).p2.z > pt.z) pt.z = (*iter).p2.z;
		}
	}
	
	return pt;
}

void TDisplayWidget::Draw_Arrow(
	QPainter							* const painter,
	const QPoint						&p1,
	const QPoint						&p2,
	int									arrow_length,
	int									arrow_width,
	const bool							start_only) const
{
	QPoint								pt;
	TVector2							line_axis;
	TVector2							line_cross_axis;
	TVector2							v,v1,v2,v3;
	
	assert(painter);
	assert(arrow_length > 0);
	assert(arrow_width > 0);
	
	pt = p2 - p1;
	
	if(pt.manhattanLength() > 0)
	{
		line_axis.Set(pt.x(),pt.y());
		line_axis.Normal();
		
		line_cross_axis.Set(-line_axis.y,line_axis.x);
		
		v2.Set(p1.x(),p1.y());
		v = v2 + line_axis * static_cast<double>(arrow_length);
		v1 = v + line_cross_axis * static_cast<double>(arrow_width);
		v3 = v - line_cross_axis * static_cast<double>(arrow_width);
		
		painter->drawLine(v1.x,v1.y,v2.x,v2.y);
		painter->drawLine(v2.x,v2.y,v3.x,v3.y);
		
		v3.Set(p2.x(),p2.y());
		painter->drawLine(v2.x,v2.y,v3.x,v3.y);
		
		if(!start_only)
		{
			v = v3 - line_axis * static_cast<double>(arrow_length);
			v1 = v + line_cross_axis * static_cast<double>(arrow_width);
			v2 = v - line_cross_axis * static_cast<double>(arrow_width);
			
			painter->drawLine(v1.x,v1.y,v3.x,v3.y);
			painter->drawLine(v2.x,v2.y,v3.x,v3.y);
		}
	}
}

void TDisplayWidget::Draw_Gauge(
	QPainter							* const painter,
	const QPoint						&p1,
	const QPoint						&p2,
	const QString						&title,
	int									gauge_width) const
{
	QPoint								pt;
	TVector2							line_axis;
	TVector2							line_cross_axis;
	TVector2							v,v1,v2,v3,v4;
	QRect								rect;
	QPoint								pts[4];
	int									radius(gauge_width / 2);
	
	assert(painter);
	
	pt = p2 - p1;

	if(pt.manhattanLength() > 0)
	{
		line_axis.Set(pt.x(),pt.y());
		line_axis.Normal();
		
		line_cross_axis.Set(-line_axis.y,line_axis.x);
		
		v.Set(p1.x(),p1.y());
		v1 = v - line_cross_axis * static_cast<double>(radius);
		v2 = v + line_cross_axis * static_cast<double>(radius);
		
		v.Set(p2.x(),p2.y());
		v3 = v + line_cross_axis * static_cast<double>(radius);
		v4 = v - line_cross_axis * static_cast<double>(radius);
		
		pts[0].setX(v1.x);
		pts[0].setY(v1.y);
		
		pts[1].setX(v2.x);
		pts[1].setY(v2.y);
		
		pts[2].setX(v3.x);
		pts[2].setY(v3.y);
		
		pts[3].setX(v4.x);
		pts[3].setY(v4.y);
		
		painter->drawPolygon(pts,4,Qt::WindingFill);
		
//		painter->drawLine(v1.x,v1.y,v2.x,v2.y);
//		painter->drawLine(v2.x,v2.y,v3.x,v3.y);
//		painter->drawLine(v3.x,v3.y,v4.x,v4.y);
//		painter->drawLine(v4.x,v4.y,v1.x,v1.y);
		
		v = (v1 + v2 + v3 + v4) / 4.0;
		
		rect = QRect(v.x - radius + 1,v.y - radius + 1,gauge_width - 1,gauge_width - 1);
		
		painter->fillRect(rect,Qt::white);
		painter->drawText(rect,Qt::AlignHCenter|Qt::AlignHCenter,title);
		
	}

}

TVector3 TDisplayWidget::Calculate_Correction(
	const TVector3 						&pos) const
{
	TVector3							correction;
	
	if(d_deflection_model)
	{
		correction = d_deflection_model->Calculate_Correction(pos);
	}
	
	return correction;
}

void TDisplayWidget::Update_Viewport(
	const TVector3						&min,
	const TVector3						&max)
{
	double                              min_x,max_x,min_y,max_y;
	double								range_y,range_z;
	TViewportMat                        W,S,V;
	static const double					MINIMUM_RANGE(0.0001);
	
	min_x = min.y;
	min_y = min.z;
	
	range_y = max.y - min.y;
	range_z = max.z - min.z;
	
	if(range_y > range_z)
	{
		max_x = range_y;
		max_y = range_y;
		
		min_y = (range_z - range_y) / 2.0;
		max_y += min_y;
	}
	else
	{
		max_x = range_z;
		max_y = range_z;
		
		min_x = (range_y - range_z) / 2.0;
		max_x += min_x;
	}
		
	if((max_y - min_y) < MINIMUM_RANGE)
	{
		max_y += (MINIMUM_RANGE / 2.0);
		min_y -= (MINIMUM_RANGE / 2.0);
	}
	
	W.tx[0] = 1;
	W.tx[1] = 0;
	W.ty[0] = 0;
	W.ty[1] = 1;
	W.to[0] = -min_x;
	W.to[1] = -min_y;
	
	S.tx[0] = (d_grid_rect.right() - d_grid_rect.left()) / (max_x - min_x);
	S.tx[1] = 0;
	S.ty[0] = 0;
	S.ty[1] = (d_grid_rect.top() - d_grid_rect.bottom()) / (max_y - min_y);
	S.to[0] = 0;
	S.to[1] = 0;
	
	V.tx[0] = 1;
	V.tx[1] = 0;
	V.ty[0] = 0;
	V.ty[1] = 1;
	V.to[0] = d_grid_rect.left();
	V.to[1] = d_grid_rect.bottom();
	
	d_viewport_mat = W;
	d_viewport_mat *=S;
	d_viewport_mat *=V;
}


