/////////////////////////////////////////////////////////////////////
//
//            X   X           X
//           XXX XX         XX
//          XXXXXXXX      XXX
//         XX X XXXXXXXXXXX
//        XXXXX XXXXXXXXX
//       XXXXX XXXXXXXXXX
//            XXX XXX XXX
//           XXX XX   XX
//           X   X     X
//
//    Copyright (C) 2003-2025  Ron Jakl
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/////////////////////////////////////////////////////////////////////

#include <QApplication>
#include <QComboBox>
#include <QGridLayout>
#include <QLabel>
#include <QLineEdit>
#include <QToolButton>
#include <QFileDialog>
#include <QSpacerItem>
#include <QStringList>
#include <QSizePolicy>
#include <QDir>

#include "../../core/xmlfile.h"

#include "optionswidget.h"

TOptionsWidget::TOptionsWidget(
	const QWidget						*parent,
	const Qt::WindowFlags				flags)
:QWidget(const_cast<QWidget*>(parent),flags)
{
	QGridLayout							*widget_layout;
	QLabel								*generator_label;
	QLabel								*requires_label;
	QLabel								*shared_libs_label;
	QSpacerItem							*widget_hspacer;
	QSpacerItem							*widget_vspacer;
	QToolButton							*shared_libs_browse_button;

	widget_layout = new QGridLayout(this);

	generator_label = new QLabel(this);
	widget_layout->addWidget(generator_label,0,0,1,1);

	d_generator_combo = new QComboBox(this);
	widget_layout->addWidget(d_generator_combo,0,1,1,1);

	requires_label = new QLabel(this);
	widget_layout->addWidget(requires_label,1,0,1,1);

	d_requires_combo = new QComboBox(this);
	widget_layout->addWidget(d_requires_combo,1,1,1,1);

	shared_libs_label = new QLabel(this);
	widget_layout->addWidget(shared_libs_label,2,0,1,1);

	d_shared_libs_edit = new QLineEdit(this);
	widget_layout->addWidget(d_shared_libs_edit,2,1,1,1);

	shared_libs_browse_button = new QToolButton(this);
	widget_layout->addWidget(shared_libs_browse_button,2,2,1,1);

	widget_hspacer = new QSpacerItem(0,0,QSizePolicy::Expanding,QSizePolicy::Minimum);
	widget_layout->addItem(widget_hspacer,0,3,4,1);

	widget_vspacer = new QSpacerItem(0,0,QSizePolicy::Minimum,QSizePolicy::Expanding);
	widget_layout->addItem(widget_vspacer,3,3,1,1);

	d_generator_combo->addItem(QStringLiteral("Using QMake"),static_cast<int>(TTypes::GENERATOR_QMAKE));
	d_generator_combo->addItem(QStringLiteral("Using CMake"),static_cast<int>(TTypes::GENERATOR_CMAKE));

	d_requires_combo->addItem(QStringLiteral("Automatically Update Requirements"),static_cast<int>(TTypes::OPTION_AUTO_UPDATE_REQUIRES));
	d_requires_combo->addItem(QStringLiteral("Manually Update Requirements"),static_cast<int>(TTypes::OPTION_MANUAL_UPDATE_REQUIRES));
	d_requires_combo->addItem(QStringLiteral("Ignore Requirements"),static_cast<int>(TTypes::OPTION_IGNORE_REQUIRES));
	
	generator_label->setText(QStringLiteral("Make File Generator:"));
	requires_label->setText(QStringLiteral("Requires Package:"));
	shared_libs_label->setText(QStringLiteral("Shared Libs Path:"));
	shared_libs_browse_button->setText(QStringLiteral("..."));
	
	connect(shared_libs_browse_button,&QToolButton::clicked,this,&TOptionsWidget::Browse_Shared_Libs_Path);
}

TOptionsWidget::~TOptionsWidget(void)
{
}

TTypes::TOptionMakeGenerator TOptionsWidget::Generator_Type(void) const
{
	return static_cast<TTypes::TOptionMakeGenerator>(d_generator_combo->currentData().toInt());
}

TTypes::TOptionRequies TOptionsWidget::Requires_Type(void) const
{
	return static_cast<TTypes::TOptionRequies>(d_requires_combo->currentData().toInt());
}

QString TOptionsWidget::Shared_Libs_Path(void) const
{
	return QDir::fromNativeSeparators(d_shared_libs_edit->text());
}

bool TOptionsWidget::Save_Defaults(
	const QString 						&file_path)
{
	QString								file_name;
	TXmlFile							xml_file;
	QDomElement							e0;
	QString								text;
	TTypes::TOptionMakeGenerator		generator_type;
	TTypes::TOptionRequies				requires_type;

	file_name = file_path + QStringLiteral("/options.xml");

	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Reset(QStringLiteral("Options"),1);

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Make_Generator"));
		
		generator_type = static_cast<TTypes::TOptionMakeGenerator>(d_generator_combo->currentData().toInt());
		
		switch(generator_type)
		{
			case TTypes::GENERATOR_QMAKE:
				text = QStringLiteral("Generator_QMake");
				break;
			case TTypes::GENERATOR_CMAKE:
				text = QStringLiteral("Generator_CMake");
				break;
		}
		xml_file.Write_Text_Node(&e0,text);

		e0 = xml_file.Create_Node(xml_file.Root_Node(),QStringLiteral("Requires_Option"));
		
		requires_type = static_cast<TTypes::TOptionRequies>(d_requires_combo->currentData().toInt());
		
		switch(requires_type)
		{
			case TTypes::OPTION_AUTO_UPDATE_REQUIRES:
				text = QStringLiteral("Requires_Auto_Update");
				break;
			case TTypes::OPTION_MANUAL_UPDATE_REQUIRES:
				text = QStringLiteral("Requires_Manual_Update");
				break;
			case TTypes::OPTION_IGNORE_REQUIRES:
				text = QStringLiteral("Requires_Ignored");
				break;
		}
		xml_file.Write_Text_Node(&e0,text);

		
		xml_file.Write_File();
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		return false;
	}

	return true;
}

bool TOptionsWidget::Load_Defaults(
	const QString 						&file_path)
{
	TXmlFile							xml_file;
	QString								file_name;
	int									version;
	bool								result(true);

	file_name = file_path + QStringLiteral("/options.xml");
	
	try
	{
		xml_file.Set_File_Name(file_name);
		xml_file.Open(QStringLiteral("Options"),&version);

		switch(version)
		{
			case 1:
				this->Load_V1(&xml_file);
				break;
				
			default:
				throw TXmlFileException(QStringLiteral("Option datafile version not recognized"));
		}
	}
	catch(TXmlFileException xml_file_exception)
	{
		d_last_error = xml_file_exception.ErrorText();
		
		this->Initialize();
		result = false;
	}
	
	return result;
}

void TOptionsWidget::Set_Generator_Type(
	const TTypes::TOptionMakeGenerator 	type)
{
	switch(type)
	{
		case TTypes::GENERATOR_QMAKE:
			d_generator_combo->setCurrentIndex(0);
			break;
			
		case TTypes::GENERATOR_CMAKE:
			d_generator_combo->setCurrentIndex(1);
			break;
	}
}

void TOptionsWidget::Set_Requires_Type(
	const TTypes::TOptionRequies 		type)
{
	switch(type)
	{
		case TTypes::OPTION_AUTO_UPDATE_REQUIRES:
			d_requires_combo->setCurrentIndex(0);
			break;
			
		case TTypes::OPTION_MANUAL_UPDATE_REQUIRES:
			d_requires_combo->setCurrentIndex(1);
			break;
			
		case TTypes::OPTION_IGNORE_REQUIRES:
			d_requires_combo->setCurrentIndex(2);
			break;
	}
}

void TOptionsWidget::Set_Shared_Libs_Path(
	const QString 						&path)
{
	d_shared_libs_edit->setText(QDir::toNativeSeparators(path));
}

void TOptionsWidget::Browse_Shared_Libs_Path(void)
{
	QString								file_path;
	QString								default_file_path;
	
	default_file_path = QDir::fromNativeSeparators(d_shared_libs_edit->text());

	file_path = QFileDialog::getExistingDirectory(this, QStringLiteral("Select Libraries Path"),default_file_path);

	if(file_path.length())
	{
		file_path = QDir::toNativeSeparators(file_path);
		
		d_shared_libs_edit->setText(file_path);
		
		emit Shared_Libs_Path_Changed(file_path);
	}
}

void TOptionsWidget::Initialize(void)
{
	d_generator_combo->setCurrentIndex(0);	// default to qmake
}

void TOptionsWidget::Load_V1(
	TXmlFile 							* const xml_file)
{
	QDomElement							e0;
	QString								text;
	TTypes::TOptionMakeGenerator		generator_type;
	TTypes::TOptionRequies				requires_type;

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Make_Generator"));
	text = xml_file->Read_Text_Node(&e0);

	if(text.compare(QStringLiteral("Generator_QMake")) == 0)
	{
		generator_type = TTypes::GENERATOR_QMAKE;
	}
	else if(text.compare(QStringLiteral("Generator_CMake")) == 0)
	{
		generator_type = TTypes::GENERATOR_CMAKE;
	}
	else
	{
		generator_type = TTypes::GENERATOR_QMAKE;
	}

	e0 = xml_file->Get_Node(xml_file->Root_Node(),QStringLiteral("Requires_Option"));
	text = xml_file->Read_Text_Node(&e0);

	if(text.compare(QStringLiteral("Requires_Auto_Update")) == 0)
	{
		requires_type = TTypes::OPTION_AUTO_UPDATE_REQUIRES;
	}
	else if(text.compare(QStringLiteral("Requires_Manual_Update")) == 0)
	{
		requires_type = TTypes::OPTION_MANUAL_UPDATE_REQUIRES;
	}
	else if(text.compare(QStringLiteral("Requires_Ignored")) == 0)
	{
		requires_type = TTypes::OPTION_IGNORE_REQUIRES;
	}
	else
	{
		requires_type = TTypes::OPTION_AUTO_UPDATE_REQUIRES;
	}

	this->Set_Generator_Type(generator_type);
	this->Set_Requires_Type(requires_type);
}
